// (c) Microsoft Corporation 2005-2007. 

#light

namespace Microsoft.FSharp.Math

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Math
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Primitives.Basics.RangeOps
open Microsoft.FSharp.Primitives.Basics.List

module LinearAlgebra =

    (*----------------------------------------------------------------------------
    !* predicates
     *--------------------------------------------------------------------------*)

    let isSymmetric a = a |> Matrix.foralli (fun i j aij -> aij = a.[j,i]) 
    let isLowerTriangular a = a |> Matrix.foralli (fun i j aij -> i>=j || aij=0.0)

    (*----------------------------------------------------------------------------
    !* choleskyFactor
     *--------------------------------------------------------------------------*)

    let choleskyFactor (a: matrix) =
      let nA,mA = a.Dimensions
      if nA<>mA              then invalid_arg "choleskyFactor: not square";
      if not (isSymmetric a) then invalid_arg "choleskyFactor: not symmetric";
      let lres = Matrix.zero nA nA (* nA=mA *)
      for j=0 to nA-1 do
        for i=j to nA-1 do (* j <= i *)
          (* Consider a_ij = sum(k=0...n-1)  (lres_ik . lresT_kj)
           *               = sum(k=0...n-1)  (lres_ik . lres_jk)
           *               = sum(k=0...j-1)  (lres_ik . lres_jk) + lres_ij . lres_jj + (0 when k>j)
           *               = psum                                + lres_ij . lres_jj
           * This determines lres_ij terms of preceeding terms.
           * lres_ij depends on lres_ik and lres_jk (and maybe lres_ii) for k<i
           *)
          let psum = sumfR (fun k -> lres.[i,k] * lres.[j,k]) (0,j-1)
          let a_ij = a.[i,j]
          if i=j then
            lres.[i,i] <- (System.Math.Sqrt (a_ij - psum))
          else
            lres.[i,j] <- ((a_ij - psum) / lres.[j,j])
        done
      done;
      // if not (isLowerTriangular lres) then failwith "choleskyFactor: not lower triangular result";
      lres

    (*----------------------------------------------------------------------------
    !* lowerTriangularInverse
     *--------------------------------------------------------------------------*)
        
    let lowerTriangularInverse (l: matrix) =
      (* Given l s.t. LT: l is lower_triangular (l_ij = 0 when i<j).
       * Finds res s.t. l.res = id *)
      let nA,mA = l.Dimensions
      let res = Matrix.zero nA nA (* nA=mA *)
      for j=0 to nA-1 do
        for i=j to nA-1 do (* j <= i *)
          (* Consider id_ij = sum(k=0...n-1)  (l_ik . res_kj)
           *                = sum(k=0...i-1)  (l_ik . res_kj) + l_ii . res_ij + (0 when k>i by LT)
           *                = psum                            + l_ii . res_ij
           * Have res_ij terms of res_kj for k<i and l_??.
           *)
          let psum   = sumfR (fun k -> l.[i,k] * res.[k,j]) (0,i-1)
          let id_ij  = if i=j then 1.0 else 0.0
          let res_ij = (id_ij - psum) / l.[i,i]
          res.[i, j] <- res_ij
        done
      done;
      res


    (*----------------------------------------------------------------------------
    !* symmetricInverse
     *--------------------------------------------------------------------------*)

    let symmetricInverse a =
      (* Given a symmetric matix.
       * Have l s.t. a = l . transpose(l)  where l is lowerTriangular.
       * Have l_inv s.t. l.l_inv = id
       * Have a_inv = transpose(l_inv).l_inv
       *)
      let l     = choleskyFactor a         
      let l_t   = l.Transpose                 
      let l_inv = lowerTriangularInverse l 
      let a_inv = l_inv.Transpose * l_inv
      a_inv


    (*----------------------------------------------------------------------------
    !* determinant 
     *--------------------------------------------------------------------------*)


    let rec upto a b = if a<= b then a::upto (a+1) b else []
    let determinant (a: matrix) =
      (* Allocates lists to manage walk over permutations.
       * Iterating through permutations a mutable array may help GC.
       *)
      let rec det js ks =
        match ks with
          | []    -> 1.0
          | k::ks -> 
            let rec split sign (preJs,js) =
              match js with
              | []    -> 0.0
              | j::js -> sign * a.[j,k] * det (List.rev preJs @ js) ks
                         +
                         split (0.0 - sign) (j::preJs,js)
            split 1.0 ([],js)
      let nA,mA = a.Dimensions
      if nA<>mA then invalid_arg "determinant: not square";
      det (upto 0 (nA-1)) (upto 0 (nA-1))

    let cholesky a =
      let l    = choleskyFactor         a
      let lInv = lowerTriangularInverse l
      l,lInv


