// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Tools.FsLex 

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Control
    open Microsoft.FSharp.Primitives.Basics
    open Microsoft.FSharp.Compatibility
    open Microsoft.FSharp.Collections

    type LexBuffer<'pos,'char> = 
        { mutable startPos: 'pos;
          mutable endPos: 'pos;
          extend_buff : unit -> unit;
          mevent : IEvent<unit>;
          meventFire : (unit -> unit) ;
          mutable buffer : 'char array;
          mutable buffer_max_scan_length : int; // number of valid charactes beyond buffer_scan_start *)
          mutable buffer_scan_start : int;      // count into the buffer when scanning *)
          mutable buffer_scan_length : int;     // number of characters scanned so far *)
          mutable lexemeLength : int;   // length of the scan at the last accepting state *)
          mutable buffer_accept_action : int;   // action related to the last accepting state *)
          mutable eof : bool; }

    module LexBufferImpl = 
        let inline blit_from_compat_array (arr1:'a []) (start1:int) (arr2: 'a array) (start2:int) (len:int) =
          for i = 0 to len - 1 do 
            (Array.set arr2 (start2+i) (CompatArray.get arr1 (start1 + i) : 'a))
          done

        let inline copy_from_compat_array (arr:'a[]) =
         let len = CompatArray.length arr in 
         let res = Array.zero_create len in
         blit_from_compat_array arr 0 res 0 len;
         res

                          
        let new_lexbuf a refill = 
            let meventFire,mevent = IEvent.create() in 
            let hole = ref None
            let res = 
                { extend_buff = (fun () -> refill (!hole).Item);
                  mevent = mevent;
                  meventFire = meventFire;
                  buffer=Array.zero_create 4096;
                  buffer_max_scan_length=0;
                  buffer_scan_start=0;
                  buffer_scan_length=0;
                  lexemeLength=0;
                  buffer_accept_action=0;
                  eof = false;
                  startPos = a ;
                  endPos = a }
            hole := Some res;
            res
            
        let code (c:char) = (# "" c : int #)

        let buffer_scan_pos (lb : LexBuffer<'pos,'char>) = lb.buffer_scan_start + lb.buffer_scan_length
        let ensure_buff (lb : LexBuffer<'pos,'char>)  n = 
            if buffer_scan_pos lb + n >= Array.length lb.buffer then 
                let repl = Array.zero_create (buffer_scan_pos lb + n) 
                Array.blit lb.buffer lb.buffer_scan_start repl lb.buffer_scan_start lb.buffer_scan_length;
                lb.buffer <- repl

        let inline from_function a f = 
            let extension=CompatArray.zero_create 4096
            new_lexbuf a (fun lb -> 
                let n = f extension (CompatArray.length extension) 
                // Printf.printf "refill from file, n = %d\n" n;*)
                ensure_buff lb n;
                blit_from_compat_array extension 0 lb.buffer (buffer_scan_pos lb) n;
                lb.buffer_max_scan_length <- lb.buffer_scan_length + n)
              
        let inline from_array a s = 
            let lb = new_lexbuf a (fun lb -> ())
            lb.buffer <- copy_from_compat_array s;
            lb.buffer_max_scan_length <- Array.length lb.buffer;
            lb
              
        // let from_text_reader (tr:System.IO.TextReader) = new_lexbuf (fun chars -> ) *)
              
        let getLexemeChar (lb : LexBuffer<'pos,'char>) n =  
            Array.get lb.buffer (n+lb.buffer_scan_start)
        let inline getLexeme (lb : LexBuffer<'pos,'char>) = 
            Array.sub lb.buffer lb.buffer_scan_start lb.lexemeLength
        let getLexemeLength (lb : LexBuffer<'pos,'char>) = lb.lexemeLength
          
        // Throw away all the input besides the lexeme *)
              
        let discardInput (lb : LexBuffer<'pos,'char>) = 
            let keep = Array.sub lb.buffer lb.buffer_scan_start lb.buffer_scan_length
            let nkeep = Array.length keep 
            Array.blit keep 0 lb.buffer 0 nkeep;
            lb.buffer_scan_start <- 0;
            lb.buffer_max_scan_length <- nkeep
                 
              
        let endOfScan (lb : LexBuffer<'pos,'char>) : int =
            // Printf.eprintf "endOfScan, lb.lexemeLength = %d\n" lb.lexemeLength;*)
            if lb.buffer_accept_action < 0 then 
                //        let inp = Array.get lb.buffer (lb.buffer_scan_start + lb.buffer_scan_length)) *)
                failwith "unrecognized input"

            //  Printf.printf "endOfScan %d state %d on unconsumed input '%c' (%d)\n" a s (Char.chr inp) inp;*)
            //   Printf.eprintf "accept, lexeme = %s\n" (lexeme lb); *)
            lb.meventFire ();
            lb.buffer_accept_action

    open LexBufferImpl
                
    type LexBuffer<'pos,'char> with
        member lb.StartPos
           with get() = lb.startPos
           and  set(b) =  lb.startPos <- b
        member lb.EndPos 
           with get() = lb.endPos
           and  set(b) =  lb.endPos <- b
        member lb.Lexeme         = getLexeme lb
        member lb.LexemeLength   = getLexemeLength lb
        member lb.LexemeChar(n)  = getLexemeChar lb n

        member lb.IsPastEndOfStream 
           with get() = lb.eof
           and  set(b) =  lb.eof <- b

        member lb.MatchEvent     = lb.mevent
        member lb.DiscardInput() = discardInput lb

        static member FromBytes    (p,arr) = (LexBufferImpl.from_array p arr : LexBuffer<'pos,byte>)
        static member FromChars    (p,arr) = (LexBufferImpl.from_array p arr : LexBuffer<'pos,char>)
        static member FromByteFunction (p,f) = (LexBufferImpl.from_function p f : LexBuffer<'pos,byte>)
        static member FromCharFunction (p,f) = (LexBufferImpl.from_function p f : LexBuffer<'pos,char>)
    end

    type AsciiTables = { trans: byte[] array; accept: byte[] }

    module AsciiTableInterpreter = 
        let sentinel = 255 * 256 + 255 
            
        let read_coded_u16 (bytes:byte[]) n = 
          let v0 = int (CompatArray.get bytes (n*2)) 
          let v1 = int (CompatArray.get bytes (n*2+1)) 
          v0 * 256 + v1


        let rec scanUntilSentinel tables s (lb : LexBuffer<_,_>) =
          // Return an endOfScan after consuming the input *)
          let a = read_coded_u16 tables.accept s 
          if a <> sentinel then 
            begin 
              lb.lexemeLength <- lb.buffer_scan_length;
              lb.buffer_accept_action <- a;
            end;
          
          if lb.buffer_scan_length = lb.buffer_max_scan_length then 
              lb.DiscardInput();
              lb.extend_buff ();
            // end of file occurs if we couldn't extend the buffer 
              if lb.buffer_scan_length = lb.buffer_max_scan_length then  
                  let snew = read_coded_u16 tables.trans.(s) 256 // == EOF 
                  if snew = sentinel then 
                    endOfScan lb 
                  else begin 
                    if lb.eof then 
                      failwith ("End of file on lexing stream");
                    lb.eof <- true;
                    // Printf.printf "state %d --> %d on eof\n" s snew;
                    scanUntilSentinel tables snew lb
                  end
              else 
                scanUntilSentinel tables s lb
          else
              // read a character - end the scan if there are no further transitions *)
              let b = Array.get (lb.buffer:byte array) (buffer_scan_pos lb)
              let inp =  int b 
              let strans = tables.trans.(s) 
              let snew = read_coded_u16 strans inp 
              if snew = sentinel then 
                endOfScan lb 
              else 
                lb.buffer_scan_length <- lb.buffer_scan_length + 1;
                // Printf.printf "state %d --> %d on '%c' (%d)\n" s snew (Char.chr inp) inp;*)
                scanUntilSentinel tables snew lb
              
              
        let interpret tables s lb =
            lb.buffer_scan_start <- lb.buffer_scan_start + lb.lexemeLength;
            lb.buffer_max_scan_length <- lb.buffer_max_scan_length - lb.lexemeLength;
            lb.buffer_scan_length <- 0;
            lb.lexemeLength <- 0;
            lb.buffer_accept_action <- -1;
            scanUntilSentinel tables s lb

    type AsciiTables 
        with 
            static member Create(trans,accept) = {trans=trans; accept=accept }
            /// Interpret tables for an ascii lexer generated by fslex. 
            member tables.Interpret(state,buf) = AsciiTableInterpreter.interpret tables state buf
        end
