// (c) Microsoft Corporation 2005-2007.

namespace Microsoft.FSharp.Text

#light

open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
open System.IO
open System.Text

/// Type of a formatting expression
///     'a : function type generated by printf
///     'b: type argument passed to %a formatters
///     'c: value generated by the overall printf action (e.g. sprint generates a string)
///     'd: value generated after post processing (e.g. failwithf generates a string internally then raises an exception)
///     'e: tuple of values generated by scan or match
type Format<'a,'b,'c,'d> =
    class
        new : value:string -> Format<'a,'b,'c,'d>
        member Value : string
    end
    
type Format<'a,'b,'c,'d,'e> = 
    class
        inherit Format<'a,'b,'c,'d>
        new: value:string -> Format<'a,'b,'c,'d,'e>
    end

/// Internal type of a formatting expression
[<System.Obsolete("Consider using Printf.Format<_,_,_,_> instead")>]
type ('a,'b,'c,'d) format4 = Format<'a,'b,'c,'d>

/// <summary> Extensible printf-style formatting for numbers and other datatypes
/// </summary>
///  <remarks>
///  <p>Format specifications are strings with "%" markers indicating format 
///   placeholders. Format placeholders consist of:</p>
///  <pre>
///    %[flags][width][.precision][type]
///  </pre> 
///  <p>where the type is interpreted as follows:</p>
///  <pre>
///     %b:         bool, formatted as "true" or "false"
///     %s:         string, formatted as its unescaped contents
///     %d, %i:     any basic integer type formatted as a decimal integer, signed if the basic integer type is signed.
///     %u:         any basic integer type formatted as an unsigned decimal integer
///     %x, %X, %o: any basic integer type formatted as an unsigned hexadecimal 
///                 (a-f)/Hexadecimal (A-F)/Octal integer
/// 
///     %e, %E, %f, %F, %g, %G: 
///                 any basic floating point type (float,float32) formatted
///                 using a C-style floating point format specifications, i.e
/// 
///     %e, %E: Signed value having the form [-]d.dddde[sign]ddd where 
///                 d is a single decimal digit, dddd is one or more decimal
///                 digits, ddd is exactly three decimal digits, and sign 
///                 is + or -
/// 
///     %f:     Signed value having the form [-]dddd.dddd, where dddd is one
///                 or more decimal digits. The number of digits before the 
///                 decimal point depends on the magnitude of the number, and 
///                 the number of digits after the decimal point depends on 
///                 the requested precision.
/// 
///     %g, %G: Signed value printed in f or e format, whichever is 
///                 more compact for the given value and precision.
/// 
/// 
///    %M:      System.Decimal value
/// 
///    %O:      Any value, printed by boxing the object and using it's ToString method(s)
/// 
///    %A:      Any value, printed by using Microsoft.FSharp.Text.StructuredFormat.Display.any_to_string with the default layout settings 
/// 
///    %a:      A general format specifier, requires two arguments:
///                 (1) a function which accepts two arguments:
///                     (a) a context parameter of the appropriate type for the
///                         given formatting function (e.g. an #System.IO.TextWriter)
///                     (b) a value to print
///                         and which either outputs or returns appropriate text.
/// 
///                 (2) the particular value to print
/// 
/// 
///    %t:      A general format specifier, requires one argument:
///                 (1) a function which accepts a context parameter of the
///                     appropriate type for the given formatting function (e.g. 
///                     an #System.IO.TextWriter)and which either outputs or returns 
///                     appropriate text.
///
///  Basic integer types are:
///     byte,sbyte,int16,uint16,int32,uint32,int64,uint64,nativeint,unativeint
///  Basic floating point types are:
///     float, float32
///
/// 
/// The following format patterns are accepted but a warning is printed:
///
///     %h(d|u|x|X|o)
///     %l(d|u|x|X|o)
///
///  The following format patterns are now deprecated:
///
///     %Ld, %Li, %Lu, %Lx, %LX, %Lo: same, but an int64
///     %nd, %ni, %nu, %nx, %nX, %no: same, but a nativeint
///     %Ud, %Ui, %Uu, %Ux, %UX, %Uo: same, but an unsigned int32 (uint32)
///     %ULd, %ULi, %ULu, %ULx, %ULX, %ULo: same, but an unsigned int64 (uint64)
///     %Und, %Uni, %Unu, %Unx, %UnX, %Uno: same, but an unsigned nativeint (unativeint)
/// </pre>
/// <p>Valid flags are: </p>
/// <pre>
///     0: add zeros instead of spaces to make up the required width
///     '-': left justify the result within the width specified
///     '+': add a '+' character if the number is positive (to match a '-' sign 
///          for negatives)
///     ' ': add an extra space if the number is positive (to match a '-' 
///              sign for negatives)
/// </pre>
/// <p>The printf '#' flag is invalid and a compile-time error will be reported if it is used.</p>
/// </remarks>
/// <examples>
/// <pre>
///   Printf.printf "Hello %s, %d = %d * %d" "World" 6 2 3;
///   Printf.sprintf "Hello %s" "World";
///   Printf.printf "Hello %a" output_string "World";
/// </pre>
/// </examples>
module Printf = 

    // REVIEW: Is this internal? And what is the documentation of it?
    type BuilderFormat<'a,'d>    = Format<'a, StringBuilder, unit, 'd>
    // REVIEW: Is this internal? And what is the documentation of it?
    type StringFormat<'a,'d>     = Format<'a, unit, string, 'd>
    // REVIEW: Is this internal? And what is the documentation of it?
    type TextWriterFormat<'a,'d> = Format<'a, TextWriter, unit, 'd>

    // These fix the final type returned
    // REVIEW: Is this internal? And what is the documentation of it?
    type BuilderFormat<'a>     = BuilderFormat<'a,unit>
    // REVIEW: Is this internal? And what is the documentation of it?
    type StringFormat<'a>      = StringFormat<'a,string>
    // REVIEW: Is this internal? And what is the documentation of it?
    type BuildStringFormat<'a> = BuilderFormat<'a,string>
    // REVIEW: Is this internal? And what is the documentation of it?
    type TextWriterFormat<'a>  = TextWriterFormat<'a,unit>

    /// Print to a string buffer, which is a StringBuilder
    val bprintf : StringBuilder ->    #BuilderFormat<'a> -> 'a

    /// Print to a text writer or an OCaml-compatible channel
    val fprintf : TextWriter -> #TextWriterFormat<'a> -> 'a

    /// Print to a text writer or an OCaml-compatible channel, adding a newline
    val fprintfn : TextWriter -> #TextWriterFormat<'a> -> 'a

    /// Print to any subtype of the .NET type System.IO.TextWriter
    val twprintf : #TextWriter -> #TextWriterFormat<'a> -> 'a

    /// Print to any subtype of the .NET type System.IO.TextWriter, and add a newline
    val twprintfn : #TextWriter -> #TextWriterFormat<'a> -> 'a

    /// Formatted printing to stderr
    val eprintf :                #TextWriterFormat<'a> -> 'a

    /// Formatted printing to stderr, adding a newline 
    val eprintfn :                #TextWriterFormat<'a> -> 'a

    /// Formatted printing to stdout
    val printf  :                #TextWriterFormat<'a> -> 'a

    /// Formatted printing to stdout, adding a newline 
    val printfn  :               #TextWriterFormat<'a> -> 'a

    /// Print to a string via an internal string buffer and return 
    /// the result as a string.  Helper printers must return strings.
    val sprintf :                #StringFormat<'a> -> 'a

    /// Print to a text writer or an OCaml-compatible channel, but intermediary 
    /// printers write via a buffer, which is often the most convenient way to write
    /// intermediary printers.  Useful if your primitive printing functions 
    /// are defined to write to buffers, but you ultimately want to
    /// go to a file.
    val bfprintf : TextWriter -> #BuilderFormat<'a> -> 'a

    /// Print to any subtype of TextWriter, but intermediary 
    /// printers write via a buffer, which is often the most convenient way to write
    /// intermediary printers.  Useful if your primitive printing functions 
    /// are defined to write to buffers, but you ultimately want to
    /// go to a file.
    val btwprintf : #TextWriter -> #BuilderFormat<'a> -> 'a

    /// Print to a string buffer, which is a StringBuilder.
    /// Return the result as a string.  Helper printers must print to 
    /// string buffers.
    val bsprintf:                #BuildStringFormat<'a> -> 'a

    /// bprintf, but call the given 'final' function to generate the result.
    /// See [[kprintf]].
    val kbprintf : (unit -> 'd)   -> StringBuilder ->    #BuilderFormat<'a,'d> -> 'a

    /// fprintf, but call the given 'final' function to generate the result.
    /// See [[kprintf]].
    val kfprintf : (unit -> 'd)   -> TextWriter -> #TextWriterFormat<'a,'d> -> 'a

    /// printf, but call the given 'final' function to generate the result.
    /// For example, these let the printing force a flush after all output has 
    /// been entered onto the channel, but not before. 
    val kprintf  : (string -> 'd) ->                #StringFormat<'a,'d> -> 'a

    /// twprintf, but call the given 'final' function to generate the result.
    /// See [[kprintf]].
    val ktwprintf : (unit -> 'd)   -> #TextWriter -> #TextWriterFormat<'a,'d> -> 'a

    /// sprintf, but call the given 'final' function to generate the result.
    /// See [[kprintf]].
    val ksprintf : (string -> 'd) ->                #StringFormat<'a,'d>  -> 'a

    /// bsprintf, but call the given 'final' function to generate the result.
    /// See [[kprintf]].
    val kbsprintf: (string -> 'd) ->                #BuilderFormat<'a,'d> -> 'a

    /// Print to a string buffer and raise an exception with the given
    /// result. Helper printers must return strings.
    val failwithf: #StringFormat<'a,'d> -> 'a

    /// Print to a string buffer and raise an exception with the given
    /// result.   Helper printers must print to string buffers.
    val bfailwithf: #BuilderFormat<'a,'d> -> 'a


    /// The type of formats accepted by print-then-call printers that write to buffers
    [<System.Obsolete("This type has been renamed to BuilderFormat")>]
    type ('a,'d) kbuffer_format  = ('a, StringBuilder,    unit,   'd) format4

    /// The type of formats accepted by print-then-call printers that write to an instance of the .NET type System.IO.TextWriter
    [<System.Obsolete("This type has been renamed to TextWriterFormat")>]
    type ('a,'d) ktextwriter_format = ('a, TextWriter, unit,   'd) format4
    /// The type of formats accepted by print-then-call printers that produce strings
    [<System.Obsolete("This type has been renamed to StringFormat")>]
    type ('a,'d) kstring_format  = ('a, unit,        string, 'd) format4

    /// The type of formats accepted by printers that write to buffers
    [<System.Obsolete("This type has been renamed to BuilderFormat")>]
    type 'a buffer_format  = ('a,unit)   kbuffer_format        
    /// The type of formats accepted by printers that write to an instance of the .NET type System.IO.TextWriter
    [<System.Obsolete("This type has been renamed to TextWriterFormat")>]
    type 'a textwriter_format = ('a,unit)   ktextwriter_format       
    /// The type of formats accepted by printers that produce strings
    [<System.Obsolete("This type has been renamed to StringFormat")>]
    type 'a string_format  = ('a,string) kstring_format        
    /// The type of formats accepted by printers that produce strings but print intermediate results to buffers 
    [<System.Obsolete("This type has been renamed to BuildStringFormat")>]
    type 'a bstring_format = ('a,string) kbuffer_format 


#if CLI_AT_LEAST_2_0
module PrintfImpl = 
    /// Harmless internal implementation type, not for use from user code
    type ('a,'b) DynCloMaker = 
        static member Make : (obj -> obj) -> obj 
#endif
