(* (c) Microsoft Corporation. All rights reserved *)

(*F#
/// The "unlinked" view of .NET metadata and code.  Central to 
///  to Abstract IL SDK.
module Microsoft.Research.AbstractIL.IL 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
F#*)

(* ====================================================================
// .NET binaries can be converted to the data structures below by using 
// the functions in the "Ilread" module. 
//
// Constituent types are listed in ascending order of complexity, 
// all the way up to the type "assembly".  Types are often specified
// via a concrete representation for the type (e.g. a record), though
// later versions of this toolkit may make these types abstract.
// Types are followed by a collection of abstract functions
// of the form "dest_XYZ" and "ABC_of_XYZ" to
// access information from objects.  Sometimes these
// abstract access functions are not complete, i.e. you may have
// to use the concrete representation directly.
//
// The second part of the file (after the definition of all the types) 
// specifies a large set of utilities for building objects belonging to 
// the types.  You will only need to become familiar with these if you 
// are transforming code or writing a code-generating compiler.
// 
// Several other utilities are also defined in this file:
//   1. A code builder for turning linear sequences of instructions 
//      augmented with exception tables into the more structured 
//      format used for code.  
//
//   2. The "typ_XYZ", "tspec_XYZ" and "mspec_XYZ" values which 
//      can be used to reference types in the "mscorlib" assembly.
//
//   3. The "rescope_XYZ" functions which can be used to lift a piece of
//      metadata from one assembly and transform it to a piece of metadata
//      suitable for use from another assembly.  The transformation adjusts
//      references in the metadata to take into account the assembly
//      where the metadata will now be located.
//
//   4. The "inst_XYZ" utilities to replace type variables
//      by types.  These are associated with generics.
//
//   5. The "intern_XYZ" tables for reducing the memory used by 
//      generated constructs.
//
//   6. The "refs_of_XYZ" utilities for finding all the assemblies 
//      referenced by a module.
//
//   7. A somewhat obscure facility to allow new instructions and types
//      to be added to the IL.  This is used by ILX.
 * ==================================================================== *)

open Nums

(*F#
/// A note on strings:  Strings in this module represent slightly 
/// different things depending on whether you are accessing the 
/// library using OCaml or a .NET language:
///   Caml: 
///      The type 'string' in this file generally represents byte arrays for
///      UTF-8 encoded strings.  
///
///   F# (and any other .NET language): 
///      The type 'string' in this file repesents a Unicode string.
///
/// Often the type "bytes" is used where we want a type that can faithfully
/// represent Unicode strings in OCaml.
 F#*)
 
type bytes = Bytes.bytes

(*F#
/// Debug info.  Values of type "source" can be attached at sequence 
/// points and some other locations.
 F#*)
 
type guid = bytes
type source_document = (*F# SourceDocument and SourceDocument = F#*) 
    { sourceLanguage: guid option; 
      sourceVendor: guid option; 
      sourceDocType: guid option; 
      sourceFile: string; }
 (*F# with 
        member Language: guid option
        member Vendor: guid option
        member DocumentType: guid option
        member File: string
      end F#*)


type source = (*F# SourceMarker and SourceMarker = F#*) 
    { sourceDocument: source_document;
      sourceLine: int;
      sourceColumn: int;
      sourceEndLine: int;
      sourceEndColumn: int }
 (*F# with 
        member Document: source_document
        member Line: int
        member Column: int
        member EndLine: int
        member EndColumn: int
      end F#*)

(*F#
/// Extensibility: ignore these unless you are generating ILX
/// structures directly.
 F#*)
 
type ext_typ (* opaque type *)
type ext_type_def_kind (* opaque type *)
type ext_instr (* opaque type *)


type assembly_name = string
type module_name = string
type locale = string
type public_key_info = (*F# PublicKey and PublicKey = F#*) 
  | PublicKey of bytes
  | PublicKeyToken of bytes
 (*F# with 
        member IsKey: bool;
        member IsKeyToken: bool;
        member Key: bytes
        member KeyToken: bytes
      end F#*)

type version_info = Nums.u16 * Nums.u16 * Nums.u16 * Nums.u16

type assembly_ref =  (*F# AssemblyRef and AssemblyRef = F#*) 
  { assemRefName: assembly_name;
    assemRefHash: bytes option;
    assemRefPublicKeyInfo: public_key_info option;
    assemRefRetargetable: bool;  (* The assembly can be retargeted (at runtime) to be from a different publisher. *)
    assemRefVersion: version_info option;
    assemRefLocale: locale option; } 
 (*F# with 
        member Name: assembly_name;
        /// The fully qualified name of the assembly reference, e.g. mscorlib, Version=1.0.3705 etc.
        member QualifiedName: string; 
        member Hash: bytes option;
        member PublicKey: public_key_info option;
        member Retargetable: bool;  
        member Version: version_info option;
        member Locale: locale option
      end F#*)

type modul_ref = (*F# ModuleRef and ModuleRef = F#*) 
  { modulRefName: module_name;
    modulRefNoMetadata: bool;
    modulRefHash: bytes option; } 
 (*F# with 
        member Name: module_name;
        member HasMetadata: bool;
        member Hash: bytes option; 
      end F#*)

(*F#
/// Scope references
///
/// Scope references are the bits of metadata attached to type names
/// that indicate where a type can be found. CIL has three 
/// kinds: local, module and assembly references:
///   o Local: the type must reside in the same module as the scope reference
///   o Module: the type must reside in the indicated module in the same
///     assembly as the scope reference
///   o Assembly: The type must reside in the indicated assembly.
///     These have no implicit context. Assembly references can end up 
///     binding to the assembly containing the reference, i.e. 
///     may be self or mutually referential.
///
///     Assembly reference may also resolve to type in an 
///     auxiliary module of an assembly when the assembly 
///     has an "exported types" (here called "classes elsewhere") table.
///
/// We represent these references by values embedded within type
/// references.  These values are usually "shared" across the data
/// structures for a module, i.e. one such value is created for each
/// assembly or module reference, and this value is reused within each
/// type object.
///
/// Note that as with method references the term structure is not 
/// _linked_, i.e. a "scope_ref" is still a _reference_ to a scope, 
/// not the scope itself.  Because the structure is not linked, 
/// the Abstract IL toolset does not require 
/// strongly connected inputs: you can manipulate an assembly
/// without loading all its dependent assemblies.  This is the primary
/// difference between Abstract IL and Reflection, and it can be both
/// a blessing and a curse depending on the kind of manipulation you
/// wish to perform.
///
/// Similarly, you can manipulate individual modules within
/// an assembly without having the whole assembly loaded.  (But note that
/// most assemblies are single-module in any case).
///
/// [scope_ref]'s _cannot_ be compared for equality in the way that
/// might be expected, in these sense that two scope_ref's may 
/// resolve to the same assembly/module even though they are not equal.  
///
///   Aside: People have suggested normalizing all scope references
///          so that this would be possible, and early versions of this
///          toolkit did this.  However, this meant that in order to load
///          each module you had to tell the toolkit which assembly it belonged to.
///          Furthermore, you had to know the exact resolved details of 
///          each assembly the module refers to.  This is
///          effectively like having a "fully-linked" view of the graph
///          of assemblies, like that provided in the Ilbind module.  This is really problematic for compile-time tools,
///          as, for example, the policy for linking at the runtime-machine
///          may actually alter the results of linking.  If such compile-time
///          assumptions are to be made then the tool built on top
///          of the toolkit rather than the toolkit itself should
///          make them.
///
/// Scope references, type references, field references and method references
/// can be "bound" to particular assemblies using the functions in "Ilbind".  
/// This simulates the resolution/binding process performed by a Common Language
/// Runtime during execution.  Various tests and derived operations
/// can then be performed on the results of binding.  See the Ilbind module
/// for more details.  Many (but not all) analyses should rightly be built on top of
/// Ilbind.
 F#*)
type scope_ref = (*F# ScopeRef and ScopeRef = F#*) 
  | ScopeRef_local (* ... be in M. *)
  | ScopeRef_module of modul_ref   (* ... be in the given module of A. *)
  | ScopeRef_assembly of assembly_ref  (* ... be in some module of the given assembly. *)
 (*F# with 
        static member Local: scope_ref
        static member Module: modul_ref -> scope_ref
        static member Assembly: assembly_ref -> scope_ref
        member IsLocalRef: bool
        member IsModuleRef: bool
        member IsAssemblyRef: bool
        member ModuleRef: modul_ref
        member AssemblyRef: assembly_ref
      end F#*)

(*F#
/// Calling conventions.  
///
/// For nearly all purposes you simply want to use CC_default combined
/// with CC_instance or CC_static, i.e.
///   instance_callconv == Callconv(CC_instance, CC_default): for an instance method
///   static_callconv   == Callconv(CC_static, CC_default): for a static method
///
/// CC_instance_explicit is only used by Managed C++, and indicates 
/// that the 'this' pointer is actually explicit in the signature. 
 F#*)

type basic_callconv = (*F# ArgumentConvention and ArgumentConvention = F#*) 
  | CC_default
  | CC_cdecl 
  | CC_stdcall 
  | CC_thiscall 
  | CC_fastcall 
  | CC_vararg
      
type hasthis = (*F# ThisConvention and ThisConvention = F#*) 
  | CC_instance           (* accepts an implicit 'this' pointer *)
  | CC_instance_explicit  (* any 'this' pointer is made explicit (C++ only) *)
  | CC_static             (* no 'this' pointer is passed *)

type callconv = (*F# CallingConvention and CallingConvention = F#*) 
  | Callconv of hasthis * basic_callconv
 (*F# with 
        member IsInstance : bool
        member IsInstanceExplicit : bool
        member IsStatic : bool
        member ThisConv : hasthis
        member BasicConv : basic_callconv
      end F#*)

val instance_callconv: callconv
val static_callconv: callconv
val is_vararg_callconv: callconv -> bool
val is_static_callconv: callconv -> bool

(*F#
/// Array shapes. For most purposes, including verification, the
/// rank is the only thing that matters.
 F#*)
 
type array_bound = i32 option 
type array_bounds = array_bound * array_bound
type array_shape = (*F# ArrayShape and ArrayShape = F#*) 
  | ArrayShape of array_bounds list (* lobound/size pairs *)
 (*F# with 
        member Rank : int
        static member SingleDimensional: ArrayShape
      end F#*)

val rank_of_array_shape: array_shape -> i32
val sdshape: array_shape (* bounds for a single dimensional, zero based array *)

(*F#
/// Type refs
 F#*)
 
type type_ref = (*F# TypeRef and TypeRef = F#*) 
   { trefScope: scope_ref;     (* Where is the type, i.e. is it in this module, in another module in this assembly or in another assembly? *)
     trefNested: string list;  (* The first of these also contains the namespace, if any *)
     trefName: string }        (* This also contains the namespace is trefNested is empty *)
 (*F# with 
        member Scope: scope_ref
        member Nesting: string list
        member Name: string
        member FullName: string
      end F#*)

val tname_of_tref: type_ref -> string
val nested_tname_of_tref: type_ref -> string
val enclosing_tnames_of_tref: type_ref -> string list
val scoref_of_tref: type_ref -> scope_ref

(*F#
/// Type specs and types.  
///
/// These are the types that appear syntactically in 
/// .NET binaries.  They can be resolved to bound types (see ilbind.ml).
///
/// Generic type definitions must be combined with
/// an instantiation to form a type.  Throughout this file, 
/// a "ref" refers to something that is uninstantiated, and
/// a "spec" to a ref that is combined with the relevant instantiations.
 F#*)
 
type type_spec = (*F# TypeSpec and TypeSpec = F#*)  
   { tspecTypeRef: type_ref;    (* Which type is being referred to? *)
     tspecInst: genactuals }    (* The type instantiation if the type is generic *)
 (*F# with 
        member TypeRef: type_ref
        member GenericArguments: genactuals
        member Scope: scope_ref
        member Nesting: string list
        member Name: string
        member FullName: string
      end F#*)

and typ = (*F# TypeSig and TypeSig = F#*)  
  | Type_void                   (* -- Used only in return and pointer types. *)
  | Type_array of array_shape * typ (* -- Array types *)
  | Type_value of type_spec     (* -- Unboxed types, including builtin types. *)
  | Type_boxed of type_spec     (* -- Reference types.  Also may be used for *)
                                (*    parents of members even if for members *)
                                (*    in value types. *)
  | Type_ptr of typ             (* -- Unmanaged pointers.  Nb. the type is *)
                                (*    used by tools and for binding *)
                                (*    only, not by the verifier. *)
  | Type_byref of typ           (* -- Managed pointers. *)
  | Type_fptr of callsig        (* -- Code pointers. *)
  | Type_tyvar of u16           (* -- Reference a generic arg. *)
  | Type_modified of            (* -- Custom modifiers. *)
        bool *                  (*   -- True if modifier is "required" *)
        type_ref *                   (*   -- The class of the custom modifier. *)
        typ                     (*   -- The type being modified. *)

  | Type_other of ext_typ        (* FOR EXTENSIONS, e.g. MS-ILX *)  

and callsig =  (*F# CallingSignature and CallingSignature = F#*)  
  { callsigCallconv: callconv;
    callsigArgs: typ list;
    callsigReturn: typ }
 (*F# with 
        member CallingConvention : CallingConvention
        member ArgumentTypes: TypeSig list
        member ReturnType: TypeSig
      end F#*)

(*F#
/// Generic parameters.  Actual generic parameters are  
/// always types.  Formal generic parameter declarations
/// may include the bounds, if any, on the generic parameter.
 F#*)
 
and genparams = genparam list
and genactuals = genactual list
and genactual = typ
and genvariance = (*F# GenericVarianceSpec and GenericVarianceSpec = F#*)  
  | NonVariant            
  | CoVariant             
  | ContraVariant         
and genparam = (*F# GenericParameterDef and GenericParameterDef = F#*)  
   { gpName: string;
     gpConstraints: typ list; 
     gpVariance: genvariance; 
     gpReferenceTypeConstraint: bool;     
     gpNotNullableValueTypeConstraint: bool;  
     gpDefaultConstructorConstraint: bool; 

 }
 (*F# with 
        member Name : string
        /// At most one is the parent type, the others are interface types 
        member Constraints: TypeSig list
        /// Variance of type parameters, only applicable to generic parameters for generic interfaces and delegates 
        member Variance: GenericVarianceSpec
        /// The type argument must be a reference type 
        member HasReferenceTypeConstraint: bool
        /// The type argument must be a value type, but not Nullable 
        member HasNotNullableValueTypeConstraint: bool
        /// The type argument must have a public nullary constructor 
        member HasDefaultConstructorConstraint: bool
      end F#*)

(*F#
/// Accessors on types
 F#*)
 
val is_array_ty: typ -> bool
val dest_array_ty: typ -> array_shape * typ

val tref_of_tspec: type_spec -> type_ref
val inst_of_tspec: type_spec -> genactuals
val tname_of_tspec: type_spec -> string
val scoref_of_tspec: type_spec -> scope_ref
val enclosing_tnames_of_tspec: type_spec -> string list

type boxity = (*F# Boxity and Boxity = F#*)  
  | AsObject
  | AsValue

val tspec_of_typ: typ -> type_spec
val boxity_of_typ: typ -> boxity
val tref_of_typ: typ -> type_ref
val is_tref_typ: typ -> bool
val inst_of_typ: typ -> genactuals
val is_tyvar_ty: typ -> bool

(*F#
/// Accessors on callsigs
 F#*)
 
val callconv_of_callsig: callsig -> callconv
val args_of_callsig: callsig -> typ list
val ret_of_callsig: callsig -> typ

(*F#
/// Formal identities of methods.  Method refs refer to methods on 
/// named types.  In general you should work with method_spec objects
/// rather than method_ref objects, because method_spec objects carry
/// information about how generic methods are instantiated.  method_ref
/// objects are only used at a few places in the Abstract IL syntax
/// and if analyzing or generating IL you will be unlikely to come across
/// these.
F#*)

type method_ref = (*F# MethodRef and MethodRef = F#*)  
   { mrefParent:type_ref;
     mrefCallconv: callconv;
     mrefArity: int; 
     mrefName: string;
     mrefArgs: typ list;
     mrefReturn: typ }
(*F# with 
       member EnclosingTypeRef: type_ref
       member CallingConvention: callconv
       member Name: string
       member GenericArity: int
       member ArgumentTypes: typ list
       member ReturnType: typ
     end F#*)

val name_of_mref: method_ref -> string
val callconv_of_mref: method_ref -> callconv
val ret_of_mref: method_ref -> typ
val args_of_mref: method_ref -> typ list
val tref_of_mref: method_ref -> type_ref 
val parent_of_mref: method_ref -> type_ref (* same as tref_of_mref *)
val genarity_of_mref: method_ref -> int
val callsig_of_mref: method_ref -> callsig
val rename_mref: string -> method_ref -> method_ref
val relocate_mref: typ -> method_ref -> method_ref

(*F#
/// Formal identities of fields.
 F#*)
 
type field_ref = (*F# FieldRef and FieldRef = F#*)  
   { frefParent: type_ref;
     frefName: string;
     frefType: typ }
(*F# with 
       member EnclosingTypeRef: type_ref
       member Name: string
       member Type: typ
     end F#*)

val typ_of_fref: field_ref -> typ
val name_of_fref: field_ref -> string
val tref_of_fref: field_ref -> type_ref

(*F#
/// Method specs and field specs
///
/// A method_spec is everything given at the callsite (apart from
/// whether the call is a tailcall and whether it is passing
/// varargs - see the instruction set below).  It is made up of 
///   1) a (possibly generic) method_ref
///   2) a "usage type" that indicates the how the type 
///      containing the declaration is being used (as
///      a value class, a boxed value class, an instantiated
///      generic class or whatever - see below)
///   3) an instantiation in the case where the method is generic.
///
/// In this unbound form of the metadata, the enclosing type may 
/// be Type_boxed even when the member is a member of a value type or
/// enumeration.  This is because the binary format of the metadata
/// does not carry enough information in a MemberRefParent to determine
/// from the binary alone whether the enclosing type is a value type or
/// not.
F#*)

type method_spec (*F# = MethodSpec and MethodSpec F#*)  
  (*F# with 
         static member Create : typ * method_ref * genactuals -> method_spec
         member MethodRef: method_ref
         member EnclosingType: typ 
         member GenericArguments: genactuals
       end F#*)
      
val dest_mspec             : method_spec -> method_ref * typ * genactuals     
val enclosing_typ_of_mspec : method_spec -> typ
val minst_of_mspec         : method_spec -> genactuals
val callconv_of_mspec      : method_spec -> callconv
val name_of_mspec          : method_spec -> string

val formal_mref_of_mspec   : method_spec -> method_ref
val formal_ret_of_mspec    : method_spec -> typ
val formal_args_of_mspec   : method_spec -> typ list
val formal_parent_of_mspec : method_spec -> type_ref
val formal_callsig_of_mspec: method_spec -> callsig

val active_inst_of_mspec   : method_spec -> genactuals
val actual_ret_of_mspec    : method_spec -> typ
val actual_args_of_mspec   : method_spec -> typ list
val actual_callsig_of_mspec: method_spec -> callsig

val genarity_of_mspec      : method_spec -> int

(*F#
/// Field specs.  The data given for a ldfld, stfld etc. instruction.
F#*)

type field_spec =  (*F# FieldSpec and FieldSpec = F#*)  
    { fspecFieldRef: field_ref;
      fspecEnclosingType: typ }    
(*F# with 
       member FieldRef: field_ref
       member EnclosingType: typ
     end F#*)

val fref_of_fspec: field_spec -> field_ref
val enclosing_typ_of_fspec: field_spec -> typ
val formal_typ_of_fspec: field_spec -> typ
val active_inst_of_fspec: field_spec -> genactuals
val actual_typ_of_fspec: field_spec -> typ
val name_of_fspec: field_spec -> string
val tref_of_fspec: field_spec -> type_ref

(*F#
/// Code labels.  In structured code each code label
/// refers to a basic block somewhere in the code of the method.
F#*)

type code_label (*F# = int F#*)


type basic_type = (*F# BasicType and BasicType = F#*)  
  | DT_R
  | DT_I1
  | DT_U1
  | DT_I2
  | DT_U2
  | DT_I4
  | DT_U4
  | DT_I8
  | DT_U8
  | DT_R4
  | DT_R8
  | DT_I
  | DT_U
  | DT_REF

type ldtoken_info = (*F# TokenSpec and TokenSpec = F#*)  
  | Token_type of typ 
  | Token_method of method_spec 
  | Token_field of field_spec

type ldc_info = (*F# ConstSpec and ConstSpec = F#*)  
  | NUM_I4 of i32
  | NUM_I8 of i64
  | NUM_R4 of ieee32
  | NUM_R8 of ieee64

type tailness = (*F# TailcallSpec and TailcallSpec = F#*)  
  | Tailcall
  | Normalcall
type alignment = (*F# AlignmentSpec and AlignmentSpec = F#*)  
  | Aligned
  | Unaligned_1
  | Unaligned_2
  | Unaligned_4
type volatility = (*F# VolatilitySpec and VolatilitySpec = F#*)   
  | Volatile
  | Nonvolatile
type readonlyByref = (*F# ReadonlySpec and ReadonlySpec = F#*)   
  | ReadonlyAddress
  | NormalAddress

type varargs = typ list option

type comparison = (*F# CompareOp and CompareOp = F#*)   
  | BI_beq        
  | BI_bge        
  | BI_bge_un     
  | BI_bgt        
  | BI_bgt_un        
  | BI_ble        
  | BI_ble_un        
  | BI_blt        
  | BI_blt_un 
  | BI_bne_un 
  | BI_brfalse 
  | BI_brtrue 

type arithmetic = (*F# ArithOp and ArithOp = F#*)   
  | AI_add    
  | AI_add_ovf
  | AI_add_ovf_un
  | AI_and    
  | AI_div   
  | AI_div_un
  | AI_ceq      
  | AI_cgt      
  | AI_cgt_un   
  | AI_clt     
  | AI_clt_un  
  | AI_conv      of basic_type
  | AI_conv_ovf  of basic_type
  | AI_conv_ovf_un  of basic_type
  | AI_mul       
  | AI_mul_ovf   
  | AI_mul_ovf_un
  | AI_rem       
  | AI_rem_un       
  | AI_shl       
  | AI_shr       
  | AI_shr_un
  | AI_sub       
  | AI_sub_ovf   
  | AI_sub_ovf_un   
  | AI_xor       
  | AI_or        
  | AI_neg       
  | AI_not       
  | AI_ldnull    
  | AI_dup       
  | AI_pop
  | AI_ckfinite 
  | AI_nop
  | AI_ldc       of basic_type * ldc_info

(*F#
/// The instruction set.                                                     
///
/// In general we don't categorize instructions, as different 
/// instruction groups are relevant for different types of operations. 
/// However we do collect the branch and compare instructions together 
/// because they all take an address, and the arithmetic ones because 
/// none of them take any direct arguments. 
F#*)
type instr = (*F# Instr and Instr = F#*)   
  (* Basic *)
  | I_arith of arithmetic
  | I_ldarg     of u16
  | I_ldarga    of u16
  | I_ldind     of alignment * volatility * basic_type
  | I_ldloc     of u16
  | I_ldloca    of u16
  | I_starg     of u16
  | I_stind     of  alignment * volatility * basic_type
  | I_stloc     of u16

  (* Control transfer *)
  | I_br    of  code_label
  | I_jmp   of method_spec
  | I_brcmp of comparison * code_label * code_label (* second label is fall-through *)
  | I_switch    of (code_label list * code_label) (* last label is fallthrough *)
  | I_ret 

   (* Method call *)
  | I_call     of tailness * method_spec * varargs
  | I_callvirt of tailness * method_spec * varargs
  | I_callconstraint of tailness * typ * method_spec * varargs
  | I_calli    of tailness * callsig * varargs
  | I_ldftn    of method_spec
  | I_newobj   of method_spec  * varargs
  
  (* Exceptions *)
  | I_throw
  | I_endfinally
  | I_endfilter
  | I_leave     of  code_label

  (* Object instructions *)
  | I_ldsfld      of volatility * field_spec
  | I_ldfld       of alignment * volatility * field_spec
  | I_ldsflda     of field_spec
  | I_ldflda      of field_spec 
  | I_stsfld      of volatility  *  field_spec
  | I_stfld       of alignment * volatility * field_spec
  | I_ldstr       of bytes  (* Beware!  This is a unicode encoding of the string! *)
  | I_isinst      of typ
  | I_castclass   of typ
  | I_ldtoken     of ldtoken_info
  | I_ldvirtftn   of method_spec

  (* Value type instructions *)
  | I_cpobj       of typ
  | I_initobj     of typ
  | I_ldobj       of alignment * volatility * typ
  | I_stobj       of alignment * volatility * typ
  | I_box         of typ
  | I_unbox       of typ
  | I_unbox_any   of typ
  | I_sizeof      of typ

  (* Generalized array instructions. In AbsIL these instructions include *)
  (* both the single-dimensional variants (with array_shape == sdshape) *)
  (* and calls to the "special" multi-dimensional "methods" such as *)
  (*   newobj void string[,]::.ctor(int32, int32) *)
  (*   call string string[,]::Get(int32, int32) *)
  (*   call string& string[,]::Address(int32, int32) *)
  (*   call void string[,]::Set(int32, int32,string) *)
  (* The IL reader transforms calls of this form to the corresponding *)
  (* generalized instruction with the corresponding array_shape *)
  (* argument. This is done to simplify the IL and make it more uniform. *)
  (* The IL writer then reverses this when emitting the binary. *)
  | I_ldelem      of basic_type
  | I_stelem      of basic_type
  | I_ldelema     of readonlyByref * array_shape * typ (* array_shape = sdshape for single dimensional arrays *)
  | I_ldelem_any  of array_shape * typ (* array_shape = sdshape for single dimensional arrays *)
  | I_stelem_any  of array_shape * typ (* array_shape = sdshape for single dimensional arrays *)
  | I_newarr      of array_shape * typ (* array_shape = sdshape for single dimensional arrays *)
  | I_ldlen

  (* "System.TypedReference" related instructions: almost *)
  (* no languages produce these, though they do occur in mscorlib.dll *)
  (* System.TypedReference represents a pair of a type and a byref-pointer *)
  (* to a value of that type. *)
  | I_mkrefany    of typ
  | I_refanytype  
  | I_refanyval   of typ
  | I_rethrow

  (* Debug-specific *)
  (* I_seqpoint is a fake instruction to represent a sequence point: *)
  (* the next instruction starts the execution of the *)
  (* statement covered by the given range - this is a *)
  (* dummy instruction and is not emitted *)
  | I_break 
  | I_seqpoint of source 

  (* Varargs - C++ only *)
  | I_arglist  

  (* Local aggregates, i.e. stack allocated data (alloca) : C++ only *)
  | I_localloc
  | I_cpblk of alignment * volatility
  | I_initblk of alignment  * volatility

  (* FOR EXTENSIONS, e.g. MS-ILX *)  
  | I_other    of ext_instr

(*F#
/// Basic Blocks
/// A basic block is a list of instructions ending in an unconditionally
/// branching instruction. A basic block has a label which must be unique
/// within the method it is located in.  Only the first instruction of
/// a basic block can be the target of a branch.
///
///   Details: The last instruction is always a control flow instruction,
///   i.e. branch, tailcall, throw etc.
/// 
///   For example
///       B1:  ldarg 1
///            pop
///            ret
///
///   will be one basic block:
///       BasicBlock("B1", [| I_ldarg(1); I_arith(AI_pop); I_ret |])
F#*)

type basic_block = (*F# BasicBlock and BasicBlock = F#*)   
    { bblockLabel: code_label;
      bblockInstrs: instr array }
 (*F# with 
        member Label : code_label
        member Instructions: instr array
      end F#*)

val label_of_bblock: basic_block -> code_label
val instrs_of_bblock: basic_block -> instr array
val destinations_of_bblock: basic_block -> code_label list
val fallthrough_of_bblock: basic_block -> code_label option
val last_of_bblock: basic_block -> instr
val instr_is_bblock_end: instr -> bool
val instr_is_tailcall: instr -> bool


(*
/// These nodes indicate a particular local variable has the given source 
/// language name within a GroupBlock. Note this does not effect local 
/// variable numbering, which is global over the whole method. 
*)
type local_debug_info = (*F# DebugMapping and DebugMapping = F#*)   
    { localNum: int;
      localName: string; }
 (*F# with 
        member LocalVarIndex : int
        member Name: string
      end F#*)

(*F#
/// Code
/// 
/// The code for a method is made up of a "code" object.  Each "code"
/// object gives the contents of the method in a "semi-structured" form, i.e.
///   1. The structure implicit in the IL exception handling tables
///      has been made explicit
///   2. No relative offsets are used in the code: all branches and
///      switch targets are made explicit as labels.
///   3. All "fallthroughs" from one basic block to the next have
///      been made explicit, by adding extra "branch" instructions to
///      the end of basic blocks which simply fallthrough to another basic
///      block.
///
/// You can convert a straight-line sequence of instructions to structured
/// code by using build_code and 
/// Most of the interesting code is contained in BasicBlocks. If you're
/// just interested in getting started with the format then begin
/// by simply considering methods which do not contain any branch 
/// instructions, or methods which do not contain any exception handling
/// constructs.
///
/// The above format has the great advantage that you can insert and 
/// delete new code blocks without needing to fixup relative offsets
/// or exception tables.  
///
/// BasicBlock(bblock)
///   See above
///
/// GroupBlock(localDebugInfo, blocks)
///   A set of blocks, with interior branching between the blocks.  For example
///       B1:  ldarg 1
///            br B2
///
///       B2:  pop
///            ret
///
///   will be two basic blocks
///       let b1 = BasicBlock("B1", [| I_ldarg(1); I_br("B2") |])
///       let b2 = BasicBlock("B2", [| I_arith(AI_pop); I_ret |])
///       GroupBlock([], [b1; b2])
///
///   A GroupBlock can include a list of debug info records for locally 
///   scoped local variables.  These indicate that within the given blocks
///   the given local variables are used for the given Debug info 
///   will only be recorded for local variables
///   declared in these nodes, and the local variable will only appear live 
///   in the debugger for the instructions covered by this node. So if you 
///   omit or erase these nodes then no debug info will be emitted for local 
///   variables.  If necessary you can have one outer ScopeBlock which specifies 
///   the information for all the local variables 
///  
///   Not all the destination labels used within a group of blocks need
///   be satisfied by that group alone.  For example, the interior "try" code
///   of "try"-"catch" construct may be:
///       B1:  ldarg 1
///            br B2
///
///       B2:  pop
///            leave B3
///
///   Again there will be two basic blocks grouped together:
///       let b1 = BasicBlock("B1", [| I_ldarg(1); I_br("B2") |])
///       let b2 = BasicBlock("B2", [| I_arith(AI_pop); I_leave("B3") |])
///       GroupBlock([], [b1; b2])
///   Here the code must be embedded in a method where "B3" is a label 
///   somewhere in the method.
///
/// RestrictBlock(labels,code) 
///   This block hides labels, i.e. the given set of labels represent
///   wiring which is purely internal to the given code block, and may not
///   be used as the target of a branch by any blocks which this block
///   is placed alongside.
///
///   For example, if a method is made up of:
///       B1:  ldarg 1
///            br B2
///
///       B2:  ret
///
///   then the label "B2" is internal.  The overall code will
///   be two basic blocks grouped together, surrounded by a RestrictBlock.
///   The label "B1" is then the only remaining visible entry to the method
///   and execution will begin at that label.
///
///       let b1 = BasicBlock("B1", [| I_ldarg(1); I_br("B2") |])
///       let b2 = BasicBlock("B2", [| I_arith(AI_pop); I_leave("B3") |])
///       let gb1 = GroupBlock([], [b1; b2])
///       RestrictBlock(["B2"], gb1)
///
///   RestrictBlock is necessary to build well-formed code.  
///
/// TryBlock(trycode,seh)
///
///   A try-catch, try-finally or try-fault block.  
///   If an exception is raised while executing
///   an instruction in 'trycode' then the exception handler given by
///   'seh' is executed.
///
/// Well-formedness conditions for code:
///
///   Well-formed code includes nodes which explicitly "hide" interior labels.
///   For example, the code object for a method may have only one entry
///   label which is not hidden, and this label will be the label where 
///   execution begins.  
///
///   Both filter and catch blocks must have one 
///   and only one entry.  These entry labels are not visible 
///   outside the filter and catch blocks. Filter has no 
///   exits (it always uses endfilter), catch may have exits. 
///   The "try" block can have multiple entries, i.e. you can branch 
///   into a try from outside.  They can have multiple exits, each of 
///   which will be a "leave".
///
F#*)
type code = (*F# Code and Code = F#*)   
  | BasicBlock of basic_block
  | GroupBlock of local_debug_info list * code list
  | RestrictBlock of code_label list * code
  | TryBlock of code * seh
and 
(*F#
///   The 'seh' specification can have several forms:
///
///     FilterCatchBlock
///       A multi-try-filter-catch block.  Execute the
///       filters in order to determine which 'catch' block to catch the
///       exception with. There are two kinds of filters - one for 
///       filtering exceptions by type and one by an instruction sequence. 
///       Note that filter blocks can't contain any exception blocks. 
///
F#*)
  seh = (*F# ExceptionBlock and ExceptionBlock = F#*)   
  | FaultBlock of code 
  | FinallyBlock of code
  | FilterCatchBlock of (filter * code) list
and filter = (*F# FilterBlock and FilterBlock = F#*)   
  | TypeFilter of typ
  | CodeFilter of code

val entries_of_code: code -> code_label list
val exits_of_code: code -> code_label list
val labels_of_code: code -> code_label list
val unique_entry_of_code: code -> code_label
val unique_exit_of_code: code -> code_label

(*F#
/// Field Init
F#*)

type field_init = (*F# FieldInit and FieldInit = F#*)   
  | FieldInit_bytes of bytes
  | FieldInit_bool of bool
  | FieldInit_char of u16
  | FieldInit_int8 of i8
  | FieldInit_int16 of i16
  | FieldInit_int32 of i32
  | FieldInit_int64 of i64
  | FieldInit_uint8 of u8
  | FieldInit_uint16 of u16
  | FieldInit_uint32 of u32
  | FieldInit_uint64 of u64
  | FieldInit_float32 of ieee32
  | FieldInit_float64 of ieee64
  | FieldInit_ref

(*F#
/// Native Types, for marshalling to the native C interface.
/// These are taken directly from the ILASM syntax, and don't really
/// correspond yet to the ECMA Spec (Partition II, 7.4).  
F#*)

type native_type = (*F# NativeType and NativeType = F#*)   
  | NativeType_empty
  | NativeType_custom of bytes * string * string * bytes (* guid,nativeTypeName,custMarshallerName,cookieString *)
  | NativeType_fixed_sysstring of i32
  | NativeType_fixed_array of i32
  | NativeType_currency
  | NativeType_lpstr
  | NativeType_lpwstr
  | NativeType_lptstr
  | NativeType_byvalstr
  | NativeType_tbstr
  | NativeType_lpstruct
  | NativeType_struct
  | NativeType_void
  | NativeType_bool
  | NativeType_int8
  | NativeType_int16
  | NativeType_int32
  | NativeType_int64
  | NativeType_float32
  | NativeType_float64
  | NativeType_unsigned_int8
  | NativeType_unsigned_int16
  | NativeType_unsigned_int32
  | NativeType_unsigned_int64
  | NativeType_array of native_type option * (i32 * i32 option) option (* optional idx of parameter giving size plus optional additive i.e. num elems *)
  | NativeType_int
  | NativeType_unsigned_int
  | NativeType_method
  | NativeType_as_any
  | (* COM interop *) NativeType_bstr
  | (* COM interop *) NativeType_iunknown
  | (* COM interop *) NativeType_idsipatch
  | (* COM interop *) NativeType_interface
  | (* COM interop *) NativeType_error               
  | (* COM interop *) NativeType_safe_array of variant_type * string option 
  | (* COM interop *) NativeType_ansi_bstr
  | (* COM interop *) NativeType_variant_bool

and variant_type = (*F# VariantType and VariantType = F#*)   
  | VariantType_empty
  | VariantType_null
  | VariantType_variant
  | VariantType_currency
  | VariantType_decimal               
  | VariantType_date               
  | VariantType_bstr               
  | VariantType_lpstr               
  | VariantType_lpwstr               
  | VariantType_iunknown               
  | VariantType_idispatch               
  | VariantType_safearray               
  | VariantType_error               
  | VariantType_hresult               
  | VariantType_carray               
  | VariantType_userdefined               
  | VariantType_record               
  | VariantType_filetime
  | VariantType_blob               
  | VariantType_stream               
  | VariantType_storage               
  | VariantType_streamed_object               
  | VariantType_stored_object               
  | VariantType_blob_object               
  | VariantType_cf                
  | VariantType_clsid
  | VariantType_void 
  | VariantType_bool
  | VariantType_int8
  | VariantType_int16                
  | VariantType_int32                
  | VariantType_int64                
  | VariantType_float32                
  | VariantType_float64                
  | VariantType_unsigned_int8                
  | VariantType_unsigned_int16                
  | VariantType_unsigned_int32                
  | VariantType_unsigned_int64                
  | VariantType_ptr                
  | VariantType_array of variant_type                
  | VariantType_vector of variant_type                
  | VariantType_byref of variant_type                
  | VariantType_int                
  | VariantType_unsigned_int                


(*F#
/// Local variables
F#*)

type local = (*F# Local and Local = F#*)   
    { localType: typ;
      localPinned: bool }
 (*F# with 
        member Type: TypeSig
        member IsPinned: bool
      end F#*)
      
val typ_of_local: local -> typ

(*F#
/// IL method bodies
F#*)

type il_method_body = (*F# ILMethodBody and ILMethodBody = F#*)   
    { ilZeroInit: bool;
      ilMaxStack: i32; (* strictly speakin should be a u16 *)
      ilNoInlining: bool;
      ilLocals: local list;
      ilCode: code;
      ilSource: source option }

val locals_of_ilmbody: il_method_body -> local list
val code_of_ilmbody: il_method_body -> code

(*F#
/// Member Access
F#*)

type member_access = (*F# MemberAccess and MemberAccess = F#*)   
  | MemAccess_assembly
  | MemAccess_compilercontrolled
  | MemAccess_famandassem
  | MemAccess_famorassem
  | MemAccess_family
  | MemAccess_private 
  | MemAccess_public 

type custom_attr_elem = (*F# AttributeElement and AttributeElement = F#*)   
  | CustomElem_string of string  option 
  | CustomElem_bool of bool
  | CustomElem_char of unichar
  | CustomElem_int8 of i8
  | CustomElem_int16 of i16
  | CustomElem_int32 of i32
  | CustomElem_int64 of i64
  | CustomElem_uint8 of u8
  | CustomElem_uint16 of u16
  | CustomElem_uint32 of u32
  | CustomElem_uint64 of u64
  | CustomElem_float32 of ieee32
  | CustomElem_float64 of ieee64
  | CustomElem_type of type_ref  
  | CustomElem_array of custom_attr_elem list

(*F#
/// Named args: values and flags indicating if they are fields or properties 
F#*)
type custom_attr_named_arg =  (string * typ * bool * custom_attr_elem)

(*F#
/// Custom attributes.  See 'decode_cattr_data' for a helper to parse the bytes 
/// to custom_attr_elem's as best as possible.  
F#*)

type custom_attr = (*F# Attribute and Attribute = F#*)   
    { customMethod: method_spec;  
      customData: bytes }
 (*F# with 
        member Data: bytes
        member Method: MethodSpec
      end F#*)

type custom_attrs (*F# = Attributes and Attributes F#*)   
(*F# with 
       member Details : custom_attr list
     end F#*)

val dest_custom_attrs: custom_attrs -> custom_attr list

(*F#
/// Method parameters and return values
F#*)

type param (*F# = Parameter and Parameter F#*)     
  = { paramName: string option;
      paramType: typ;
      paramDefault: field_init option;  
      paramMarshal: native_type option; (* Marshalling map for parameters. COM Interop only. *)
      paramIn: bool;
      paramOut: bool;
      paramOptional: bool;
      paramCustomAttrs: custom_attrs }
 (*F# with 
        member Name: string option
        member Type: TypeSig
        member Default: FieldInit option 
        member Marshal: NativeType option 
        member IsIn: bool
        member IsOut: bool
        member IsOptional: bool
        member CustomAttrs: custom_attrs;
      end F#*)

val name_of_param: param -> string option
val typ_of_param: param -> typ

(*F#
/// Method return values
F#*)

type returnv = (*F# ReturnValue and ReturnValue = F#*)  
    { returnMarshal: native_type option;
      returnType: typ;
      returnCustomAttrs : custom_attrs; }
 (*F# with 
        member Type: TypeSig
        member Marshal: NativeType option 
        member CustomAttrs: custom_attrs;
      end F#*)

val typ_of_return: returnv -> typ

(*F#
/// Security Permissions
/// 
/// Attached to various structures...
F#*)

type security_action = (*F# SecurityAction and SecurityAction = F#*)  
  | SecAction_request 
  | SecAction_demand
  | SecAction_assert
  | SecAction_deny
  | SecAction_permitonly
  | SecAction_linkcheck 
  | SecAction_inheritcheck
  | SecAction_reqmin
  | SecAction_reqopt
  | SecAction_reqrefuse
  | SecAction_prejitgrant
  | SecAction_prejitdeny
  | SecAction_noncasdemand
  | SecAction_noncaslinkdemand
  | SecAction_noncasinheritance
  | SecAction_linkdemandchoice
  | SecAction_inheritancedemandchoice
  | SecAction_demandchoice

type permission_value = (*F# PermissionValue and PermissionValue = F#*)  
  | PermissionValue_bool of bool
  | PermissionValue_int32 of i32
  | PermissionValue_string of string
  | PermissionValue_enum_int8 of type_ref * i8
  | PermissionValue_enum_int16 of type_ref * i16
  | PermissionValue_enum_int32 of type_ref * i32

(* @todo factor this differently - ILDASM currently doesn't expose *)
(* the contents of a security permission set in detail *)
type permission = (*F# Permission and Permission = F#*)  
  | Permission of security_action * typ * (string * permission_value) list
  | PermissionSet of security_action * bytes

(*F# 
/// Abstract type equivalent to permission list - use helpers 
/// below to construct/destruct these 
F#*)
type security_decls (*F# = Permissions and Permissions F#*)  
(*F# with 
      member Details : permission list 
     end F#*)

val dest_security_decls: security_decls -> permission list

(*F#
/// PInvoke attributes.
F#*)

type pinvokeCallConv (*F# = PInvokeCallingConvention and PInvokeCallingConvention F#*) = 
  | PInvokeCallConvNone
  | PInvokeCallConvCdecl
  | PInvokeCallConvStdcall
  | PInvokeCallConvThiscall
  | PInvokeCallConvFastcall
  | PInvokeCallConvWinapi

type pinvokeEncoding (*F# = PInvokeCharEncoding and PInvokeCharEncoding F#*)  = 
  | PInvokeEncodingNone
  | PInvokeEncodingAnsi
  | PInvokeEncodingUnicode
  | PInvokeEncodingAuto

type pinvokeBestFit  (*F# = PInvokeCharBestFit and PInvokeCharBestFit F#*) =
  | PInvokeBestFitUseAssem
  | PInvokeBestFitEnabled
  | PInvokeBestFitDisabled

type pinvokeThrowOnUnmappableChar (*F# = PInvokeThrowOnUnmappableChar and PInvokeThrowOnUnmappableChar F#*)  =
  | PInvokeThrowOnUnmappableCharUseAssem
  | PInvokeThrowOnUnmappableCharEnabled
  | PInvokeThrowOnUnmappableCharDisabled

type pinvoke_method (*F# = PInvokeMethod and PInvokeMethod F#*)  =  
    { pinvokeWhere: modul_ref;
      pinvokeName: string;
      pinvokeCallconv: pinvokeCallConv;
      pinvokeEncoding: pinvokeEncoding;
      pinvokeNoMangle: bool;
      pinvokeLastErr: bool;
      pinvokeThrowOnUnmappableChar: pinvokeThrowOnUnmappableChar;
      pinvokeBestFit: pinvokeBestFit }
 (*F# with 
        member Where: ModuleRef
        member Name: string
        member CallingConvention: PInvokeCallingConvention
        member CharEncoding: PInvokeCharEncoding
        member NoMangle: bool
        member LastError: bool
        member ThrowOnUnmappableChar: PInvokeThrowOnUnmappableChar
        member CharBestFit: PInvokeCharBestFit
      end F#*)


(*F#
/// [overrides_spec] - refer to a method declaration in a superclass 
/// or superinterface. Used for overriding/method impls.  Includes
/// a type for the parent for the same reason that a method specs
/// includes the type of the enclosing type, i.e. the type
/// gives the "genactuals" at which the parent type is being used.
F#*)

type overrides_spec (*F# = OverridesSpec and OverridesSpec F#*)
    = OverridesSpec of method_ref * typ
 (*F# with 
        member MethodRef: MethodRef
        member EnclosingType: typ 
      end F#*)

val enclosing_typ_of_ospec: overrides_spec -> typ
val formal_mref_of_ospec: overrides_spec -> method_ref
val callconv_of_ospec: overrides_spec -> callconv
val formal_ret_of_ospec: overrides_spec -> typ
val formal_args_of_ospec: overrides_spec -> typ list
val tref_of_ospec: overrides_spec -> type_ref
val name_of_ospec: overrides_spec -> string
val formal_callsig_of_ospec: overrides_spec -> callsig
val actual_callsig_of_ospec: overrides_spec -> callsig

(*F#
/// virtOverrides is for ILX only.  It gives the signature of
/// the method this method_def overrides. The entry can always be None 
/// for non-ILX code, and can normally be None even for most ILX code.
/// None indicates that the overridden method is inferred during 
/// method table layout/verification.  None can be used when 
/// the signature and name of the method being overriden is absolutely
/// identical to the signature of the method that 
/// does the overriding (this is always the case for non-ILX code). 
/// This field should be 'Some' if the signatures are not identical
/// due to generic inheritance AND you are using ILX to implement generics.
F#*)

type virtual_info (*F# = MethodVirtualInfo and MethodVirtualInfo F#*) =  
    { virtFinal: bool; 
      virtNewslot: bool; 
      virtStrict: bool; (* mdCheckAccessOnOverride *)
      virtAbstract: bool;
      virtOverrides: overrides_spec option; }
 (*F# with 
        member IsFinal: bool
        member IsNewSlot: bool
        member IsCheckAccessOnOverride: bool
        member IsAbstract: bool
        member OverridenMethod: overrides_spec option 
      end F#*)


type method_kind (*F# = MethodKind and MethodKind F#*) =  
  | MethodKind_static 
  | MethodKind_cctor 
  | MethodKind_ctor 
  | MethodKind_nonvirtual 
  | MethodKind_virtual of virtual_info

type method_body_details (*F# = MethodBody and MethodBody F#*) = 
  | MethodBody_il of il_method_body
  | MethodBody_pinvoke of pinvoke_method       (* platform invoke to native  *)
  | MethodBody_abstract
  | MethodBody_native

type method_code_kind  (*F# = MethodCodeKind and MethodCodeKind F#*) =
  | MethodCodeKind_il
  | MethodCodeKind_native
  | MethodCodeKind_runtime

type method_body  (*F# = LazyMethodBody and LazyMethodBody F#*)  (* isomorphic to method_body_details *)
(*F# with 
       member Details : method_body_details 
     end F#*)

val dest_mbody : method_body -> method_body_details 

(*F#
/// Method definitions.
///
/// There are several different flavours of methods (constructors,
/// abstract, virtual, static, instance, class constructors).  There
/// is no perfect factorization of these as the combinations are not
/// independent.  
F#*)

type method_def = (*F# MethodDef and MethodDef = F#*)   
    { mdName: string;
      mdKind: method_kind;
      mdCallconv: callconv;
      mdParams: param list;
      mdReturn: returnv;
      mdAccess: member_access;
      mdBody: method_body;   
      mdCodeKind: method_code_kind;   
      mdInternalCall: bool;
      mdManaged: bool;
      mdForwardRef: bool;
      mdSecurityDecls: security_decls;
      mdHasSecurity: bool; (* Note: some methods are marked "HasSecurity" even if there are no permissions attached, e.g. if they use SuppressUnmanagedCodeSecurityAttribute *)
      mdEntrypoint:bool;
      mdReqSecObj: bool;
      mdHideBySig: bool;
      mdSpecialName: bool;
      mdUnmanagedExport: bool; (* -- The method is exported to unmanaged code using COM interop. *)
      mdSynchronized: bool;
      mdPreserveSig: bool;
      mdMustRun: bool; (* Whidbey feature: SafeHandle finalizer must be run *)
      mdExport: (i32 * string option) option; 
      mdVtableEntry: (i32 * i32) option;
     
      mdGenericParams: genparams;
      mdCustomAttrs: custom_attrs; }
 (*F#
    with 
      member Name: string;
      //mdKind: method_kind;
      //Body: method_body;   
      //CodeKind: method_code_kind;   
      member CallingConvention: callconv;
      member Parameters: param list;
      member Return: returnv;
      member Access: member_access;
      member IsInternalCall: bool;
      member IsManaged: bool;
      member IsForwardRef: bool;
      member SecurityDecls: security_decls;
      /// Note: some methods are marked "HasSecurity" even if there are no permissions attached, e.g. if they use SuppressUnmanagedCodeSecurityAttribute 
      member HasSecurity: bool; 
      member IsEntrypoint:bool;
      member IsReqSecObj: bool;
      member IsHideBySig: bool;
      /// The method is exported to unmanaged code using COM interop. 
      member IsUnmanagedExport: bool; 
      member IsSynchronized: bool;
      member IsPreserveSig: bool;
      /// Whidbey feature: SafeHandle finalizer must be run 
      member IsMustRun: bool; 
      //member Export: (i32 * string option) option; 
      //member VtableEntry: (i32 * i32) option;  
      member GenericParams: genparams;
      member CustomAttrs: custom_attrs;
      member IsIL : bool
      member Code : code option
      member Locals : local list
      member IsNoInline : bool
      member MaxStack : i32
      member IsZeroInit : bool
      
      /// .cctor methods.  The predicates (IsClassInitializer,IsConstructor,IsStatic,IsNonVirtualInstance,IsVirtual) form a complete, non-overlapping classification of this type
      member IsClassInitializer: bool
      /// .ctor methods.  The predicates (IsClassInitializer,IsConstructor,IsStatic,IsNonVirtualInstance,IsVirtual) form a complete, non-overlapping classification of this type
      member IsConstructor: bool
      /// static methods.  The predicates (IsClassInitializer,IsConstructor,IsStatic,IsNonVirtualInstance,IsVirtual) form a complete, non-overlapping classification of this type
      member IsStatic: bool
      /// instance methods that are not virtual.  The predicates (IsClassInitializer,IsConstructor,IsStatic,IsNonVirtualInstance,IsVirtual) form a complete, non-overlapping classification of this type
      member IsNonVirtualInstance: bool
      /// instance methods that are virtual or abstract or implement an interface slot.  The predicates (IsClassInitializer,IsConstructor,IsStatic,IsNonVirtualInstance,IsVirtual) form a complete, non-overlapping classification of this type
      member IsVirtual: bool
      
      member IsFinal: bool
      member IsNewSlot: bool
      member IsCheckAccessOnOverride : bool
      member IsAbstract: bool
    end
  F#*)
  
val name_of_mdef: method_def -> string
val num_args: method_def -> int
val mdef_is_virt: method_def -> bool
val mdef_is_il: method_def -> bool
val params_of_mdef: method_def -> param list
val callconv_of_mdef: method_def -> callconv
val ilmbody_of_mdef: method_def -> il_method_body
val code_of_mdef: method_def -> code
val entry_of_mdef: method_def -> code_label
val callsig_of_mdef: method_def -> callsig

val qualified_name_of_aref : assembly_ref -> string
val qualified_name_of_scoref : scope_ref -> string
val qualified_name_of_tref: type_ref -> string

(*F#
/// Tables of methods.  Logically equivalent to a list of methods but
/// the table is kept in a form optimized for looking up methods by 
/// name and arity.
F#*)

type methods (*F# = MethodDefs and MethodDefs F#*)  
 (* abstract type equivalent to [method_def list] *)
(*F# with 
       member Details : method_def list 
     end F#*)

val dest_mdefs: methods -> method_def list
val filter_mdefs: (method_def -> bool) -> methods -> methods
val find_mdefs_by_name: string -> methods -> method_def list
val find_mdefs_by_arity: string * int -> methods -> method_def list

(*F#
/// Field definitions
F#*)

type field_def = (*F# FieldDef and FieldDef = F#*)   
    { fdName: string;
      fdType: typ;
      fdStatic: bool;
      fdAccess: member_access;
      fdData:  bytes option;
      fdInit: field_init option;  
      fdOffset:  i32 option; 
      fdSpecialName: bool;
      fdMarshal: native_type option; 
      fdNotSerialized: bool;
      fdLiteral: bool ;
      fdInitOnly: bool;
      fdCustomAttrs: custom_attrs; }
(*F# with 
      member Name: string;
      member Type: typ;
      member IsStatic: bool;
      member Access: member_access;
      member Data:  bytes option;
      member LiteralValue: field_init option;  
      /// The explicit offset in bytes when explicit layout is used.
      member Offset:  i32 option; 
      member Marshal: native_type option; 
      member NotSerialized: bool;
      member IsLiteral: bool ;
      member IsInitOnly: bool;
      member CustomAttrs: custom_attrs;
     end F#*)

val typ_of_fdef : field_def -> typ
val name_of_fdef: field_def -> string

(*F#
/// Tables of fields.  Logically equivalent to a list of fields but
/// the table is kept in a form optimized for looking up fields by 
/// name.
F#*)

type fields (*F# = FieldDefs and FieldDefs F#*) 
(*F# with 
       member Details : field_def list 
     end F#*)

val dest_fdefs: fields -> field_def list
val filter_fdefs: (field_def -> bool) -> fields -> fields
val find_fdefs: string -> fields -> field_def list

(*F#
/// Event definitions
F#*)

type event_def (*F# = EventDef and EventDef F#*) = 
    { eventType: typ option; 
      eventName: string;
      eventRTSpecialName: bool;
      eventSpecialName: bool;
      eventAddOn: method_ref; 
      eventRemoveOn: method_ref;
      eventFire: method_ref option;
      eventOther: method_ref list;
      eventCustomAttrs: custom_attrs; }
 (*F#
    with 
      member Type: typ option; 
      member Name: string;
      member AddMethod: method_ref; 
      member RemoveMethod: method_ref;
      member FireMethod: method_ref option;
      member OtherMethods: method_ref list;
      member CustomAttrs: custom_attrs; 
    end
  F#*)

(*F#
/// Table of those events in a type definition.
F#*)

type events (*F# = EventDefs and EventDefs F#*)
(*F# with 
       member Details : event_def list 
     end F#*)

val dest_edefs: events -> event_def list
val filter_edefs: (event_def -> bool) -> events -> events
val find_edefs: string -> events -> event_def list

(*F#
/// Property definitions
F#*)

type property_def (*F# = PropertyDef and PropertyDef F#*) = 
    { propName: string;
      propRTSpecialName: bool;
      propSpecialName: bool;
      propSet: method_ref option;
      propGet: method_ref option;
      propCallconv: hasthis;
      propType: typ;          
      propInit: field_init option;
      propArgs: typ list;
      propCustomAttrs: custom_attrs; }
  (*F#
    with
      member Name: string;
      member SetMethod: method_ref option;
      member GetMethod: method_ref option;
      member CallingConvention: hasthis;
      member Type: typ;          
      member Init: field_init option;
      member Args: typ list;
      member CustomAttrs: custom_attrs; 
     end F#*)

(*F#
/// Table of those properties in a type definition.
F#*)

type properties (*F# = PropertyDefs and PropertyDefs F#*)
(*F# with 
       member Details : property_def list 
     end F#*)

val dest_pdefs: properties -> property_def  list
val filter_pdefs: (property_def -> bool) -> properties -> properties
val find_pdefs: string -> properties -> property_def list

(*F#
/// Method Impls
///
/// If there is an entry (pms --&gt; ms) in this table, then method [ms] 
/// is used to implement method [pms] for the purposes of this class 
/// and its subclasses. 
F#*)

type method_impl (*F# = MethodImplDef and MethodImplDef F#*)= 
   { mimplOverrides: overrides_spec;
     mimplOverrideBy: method_spec }

type method_impls (*F# = MethodImplDefs and MethodImplDefs F#*)
(*F# with 
       member Details : method_impl list 
     end F#*)

val dest_mimpls: method_impls -> method_impl list

(*F#
/// Type Layout information
F#*)

type type_layout (*F# = TypeDefLayout and TypeDefLayout F#*)=
  | TypeLayout_auto
  | TypeLayout_sequential of type_layout_info
  | TypeLayout_explicit of type_layout_info 

and type_layout_info (*F# = TypeDefLayoutInfo and TypeDefLayoutInfo F#*)=
    { typeSize: i32 option;
      typePack: u16 option } 
 (*F# with 
        member Size: i32 option
        member Pack: u16 option 
      end F#*)

(*F#
/// Type init semantics
F#*)
type type_init (*F# = TypeDefInitSemantics and TypeDefInitSemantics F#*)=
  | TypeInit_beforefield
  | TypeInit_beforeany

(*F#
/// Default Unicode encoding for P/Invoke  within a type
F#*)
type type_encoding (*F# = TypeDefDefaultPInvokeEncoding and TypeDefDefaultPInvokeEncoding F#*)= 
  | TypeEncoding_ansi
  | TypeEncoding_autochar
  | TypeEncoding_unicode

(*F#
/// Type Access
F#*)
type type_access  (*F# = TypeDefAccess and TypeDefAccess F#*)=  
  | TypeAccess_public 
  | TypeAccess_private
  | TypeAccess_nested of member_access 

(*F#
/// A categorization of type definitions into "kinds"
F#*)
(*-------------------------------------------------------------------
 *
 * A note for the nit-picky.... In theory, the "kind" of a type 
 * definition can only be  partially determined prior to binding.  
 * For example, you cannot really, absolutely tell if a type is 
 * really, absolutely a value type until you bind the 
 * super class and test it for type equality against System.ValueType.  
 * However, this is unbearably annoying, as it means you 
 * have to load "mscorlib" and perform bind operations 
 * in order to be able to determine some quite simple 
 * things.  So we approximate by simply looking at the name
 * of the superclass when loading.
 * ------------------------------------------------------------------ *)

type type_def_kind (*F# = TypeDefKind and TypeDefKind F#*)=  
  | TypeDef_class
  | TypeDef_valuetype
  | TypeDef_interface
  | TypeDef_enum 
  | TypeDef_delegate 
  (* FOR EXTENSIONS, e.g. MS-ILX *)  
  | TypeDef_other of ext_type_def_kind

(* ------------------------------------------------------------------ 
 * Type Names
 *
 * The name of a type stored in the tdName field is as follows:
 *   - For outer types it is, for example, System.String, i.e.
 *     the namespace followed by the type name.
 *   - For nested types, it is simply the type name.  The namespace
 *     must be gleaned from the context in which the nested type
 *     lies.
 * ------------------------------------------------------------------ *)

type namespace_and_typename = string list * string

(*F#
/// The [split_type_name] utility helps you split a string representing
/// a type name into the leading namespace elements (if any), the
/// names of any nested types and the type name itself.  This function
/// memoizes and interns the splitting of the namespace portion of
/// the type name. 
F#*)
val split_namespace: string -> string list
val split_namespace_memoized: string -> string list
val split_type_name: string -> namespace_and_typename

(*F#
/// Type Definitions 
///
/// As for methods there are several important constraints not encoded 
/// in the type definition below, for example that the super class of
/// an interface type is always None, or that enumerations always
/// have a very specific form.
F#*)
type type_def =  (*F# TypeDef and TypeDef = F#*)   
    { tdKind: type_def_kind;
      tdName: string;  
      tdGenericParams: genparams;  
      tdAccess: type_access;  
      tdAbstract: bool;
      tdSealed: bool; 
      tdSerializable: bool; 
      tdComInterop: bool; (* Class or interface generated for COM interop *) 
      tdLayout: type_layout;
      tdSpecialName: bool;
      tdEncoding: type_encoding;
      tdNested: types;
      tdImplements: typ list;  
      tdExtends: typ option; 
      tdMethodDefs: methods;
      tdSecurityDecls: security_decls;
      tdHasSecurity: bool; (* Note: some classes are marked "HasSecurity" even if there are no permissions attached, e.g. if they use SuppressUnmanagedCodeSecurityAttribute *)
      tdFieldDefs: fields;
      tdMethodImpls: method_impls;
      tdInitSemantics: type_init;
      tdEvents: events;
      tdProperties: properties;
      tdCustomAttrs: custom_attrs; }
 (*F# with 
      member IsClass: bool;
      member IsValueType: bool;
      member IsInterface: bool;
      member IsEnum: bool;
      member IsDelegate: bool;
      member Name: string;  
      member GenericParams: genparams;  
      member Access: type_access;  
      member IsAbstract: bool;
      member Sealed: bool; 
      member IsSerializable: bool; 
      /// Class or interface generated for COM interop 
      member IsComInterop: bool; 
      member Layout: type_layout;
      member IsSpecialName: bool;
      member Encoding: type_encoding;
      member NestedTypes: types;
      member Implements: typ list;  
      member Extends: typ option; 
      member SecurityDecls: security_decls;
      /// Note: some classes are marked "HasSecurity" even if there are no permissions attached, e.g. if they use SuppressUnmanagedCodeSecurityAttribute 
      member HasSecurity: bool; 
      member Fields: fields;
      member Methods: methods;
      member MethodImpls: method_impls;
      member Events: events;
      member Properties: properties;
      member InitSemantics: type_init;
      member CustomAttrs: custom_attrs;
      end F#*)



(*F#
/// Tables of named type definitions.  The types and table may contain on-demand
/// (lazy) computations, e.g. the actual reading of some aspects
/// of a type definition may be delayed if the reader being used supports
/// this.
///
/// This is an abstract type equivalent to "type_def list" 
F#*) 
and types (*F# = TypeDefs and TypeDefs F#*) 
(*F# with 
       member Details : type_def list 
     end F#*)

val kind_of_tdef: type_def -> type_def_kind
val name_of_tdef: type_def -> string
val gparams_of_tdef: type_def -> genparams
val fields_of_tdef: type_def -> fields
val methods_of_tdef: type_def -> methods
val mimpls_of_tdef: type_def -> method_impls
val properties_of_tdef: type_def -> properties
val events_of_tdef: type_def -> events
val nested_of_tdef: type_def -> types
val extends_of_tdef: type_def -> typ option
val implements_of_tdef: type_def -> typ list
val access_of_tdef: type_def -> type_access
val layout_of_tdef: type_def -> type_layout
val sealed_of_tdef: type_def -> bool
val abstract_of_tdef: type_def -> bool
val serializable_of_tdef: type_def -> bool
val custom_attrs_of_tdef: type_def -> custom_attrs
val security_decls_of_tdef: type_def -> security_decls
val encoding_of_tdef: type_def -> type_encoding
val initsemantics_of_tdef: type_def -> type_init

val is_enum_tdef: type_def -> bool
val is_value_tdef: type_def -> bool
val name_of_nested_tdef: type_def list * type_def -> string

(*F#
/// Find the method definition corresponding to the given property or 
/// event operation. These are always in the same class as the property 
/// or event. This is useful especially if your code is not using the Ilbind 
/// API to bind references. 
F#*)
val mdef_for_semantic_mref: type_def -> method_ref -> method_def

val iter_tdefs: (type_def -> unit) -> types -> unit
val dest_tdefs: types -> type_def  list
(*F#
/// Calls to [find_tdef] will result in any laziness in the overall 
/// set of types being read in in addition 
/// to the details for the type found, but the remaining individual 
/// type definitions will not be read. 
F#*)
val find_tdef: string -> types -> type_def

val dest_lazy_tdefs: types -> (string list * string * custom_attrs * type_def Lazy.t) list

val tname_for_toplevel: string
val is_toplevel_tname: string -> bool

val ungenericize_tname: string -> string (* e.g. List`1 --> List *)

(* -------------------------------------------------------------------- 
 * "Classes Elsewhere" - classes in auxiliary modules.
 *
 * Manifests include declarations for all the classes in an 
 * assembly, regardless of which module they are in.
 *
 * The ".class extern" construct describes so-called exported types -- 
 * these are public classes defined in the auxiliary modules of this assembly,
 * i.e. modules other than the manifest-carrying module. 
 * 
 * For example, if you have a two-module 
 * assembly (A.DLL and B.DLL), and the manifest resides in the A.DLL, 
 * then in the manifest all the public classes declared in B.DLL should
 * be defined as exported types, i.e., as ".class extern". The public classes 
 * defined in A.DLL should not be defined as ".class extern" -- they are 
 * already available in the manifest-carrying module. The union of all 
 * public classes defined in the manifest-carrying module and all 
 * exported types defined there is the set of all classes exposed by 
 * this assembly. Thus, by analysing the metadata of the manifest-carrying 
 * module of an assembly, you can identify all the classes exposed by 
 * this assembly, and where to find them.
 *
 * Nested classes found in external modules should also be located in 
 * this table, suitably nested inside another "exported_type"
 * definition.
 * -------------------------------------------------------------------- *)

(* these are only found in the "Nested" field of exported_type objects *)
type nested_exported_type (*F# = NestedExportedType and NestedExportedType F#*) = 
    { nestedExportedTypeName: string;
      nestedExportedTypeAccess: member_access;
      nestedExportedTypeNested: nested_exported_types;
      nestedExportedTypeCustomAttrs: custom_attrs } 

and nested_exported_types(*F# = NestedExportedTypes and NestedExportedTypes F#*)
(*F# with 
       member Details :  nested_exported_type list
     end F#*)

(* these are only found in the exported_types table in the manifest *)
type exported_type (*F# = ExportedType and ExportedType F#*)  = 
    { exportedTypeScope: scope_ref;
      exportedTypeName: string;
      exportedTypeForwarder: bool;
      exportedTypeAccess: type_access;
      exportedTypeNested: nested_exported_types;
      exportedTypeCustomAttrs: custom_attrs } 
 (*F# with 
        member ModuleRef: scope_ref
        member Name: string
        member Access: type_access
        member Nested: nested_exported_types
        member CustomAttrs: custom_attrs 
      end F#*)

type exported_types (*F# = ExportedTypes and ExportedTypes F#*)
(*F# with 
       member Details : exported_type list
     end F#*)

val dest_nested_exported_types: nested_exported_types -> nested_exported_type  list
val dest_exported_types: exported_types -> exported_type list
val find_exported_type: string -> exported_types -> exported_type

(* -------------------------------------------------------------------- 
 * "Manifest Resources" are chunks of resource data, being one of:
 *   - the data section of the current module (bytes of resource given directly) 
 *   - in an external file in this assembly (offset given in the resource_where field) 
 *   - as a resources in another assembly of the same name.  
 * -------------------------------------------------------------------- *)

type resource_access (*F# = ResourceAccess and ResourceAccess F#*) = 
  | Resource_public 
  | Resource_private 
type resource_where (*F# = ResourceLocation and ResourceLocation F#*) = 
  | Resource_local of (unit -> bytes)  (* resources may be re-read each time this funciton is called *)
  | Resource_file of modul_ref * int32
  | Resource_assembly of assembly_ref

type resource (*F# = Resource and Resource F#*) = 
    { resourceName: string;
      resourceWhere: resource_where;
      resourceAccess: resource_access;
      resourceCustomAttrs: custom_attrs }
 (*F# with 
        member Name: string
        member Location: ResourceLocation
        member Access: ResourceAccess
        member CustomAttrs: custom_attrs 
      end F#*)

(* -------------------------------------------------------------------- 
 * Table of resources in a module
 * -------------------------------------------------------------------- *)

type resources (*F# = Resources and Resources F#*)
(*F# with 
       member Details : resource list 
     end F#*)

val dest_resources: resources -> resource  list

(* -------------------------------------------------------------------- 
 * Fixups are pretty obscure stuff for C++ code.  These are not
 * yet correctly represented in the AbstractIL syntax.
 * -------------------------------------------------------------------- *)

(* type fixup = Fixup of (i32 * string list * data_label) *)
(* type fixups = fixup list *)

(* -------------------------------------------------------------------- 
 * Manifests, The "main" module of an assembly, and Assemblies. 
 * 
 * The main module of an assembly is a module plus some manifest information.
 *
 * An assembly is built by joining together a "main" module plus 
 * several auxiliary modules. 
 * -------------------------------------------------------------------- *)

type longevity (*F# = Longevity and  Longevity F#*) = 
  | LongevityUnspecified
  | LongevityLibrary
  | LongevityPlatformAppDomain
  | LongevityPlatformProcess
  | LongevityPlatformSystem

type manifest = (*F# Manifest and Manifest = F#*)   
    { manifestName: string;
      manifestAuxModuleHashAlgorithm: i32; 
      manifestSecurityDecls: security_decls;
      manifestPublicKey: bytes option;  
      manifestVersion: version_info option;
      manifestLocale: locale option;
      manifestCustomAttrs: custom_attrs;
      manifestLongevity: longevity; 
      manifestDisableJitOptimizations: bool;
      manifestJitTracking: bool;
      manifestExportedTypes: exported_types;
      manifestEntrypointElsewhere: modul_ref option;
    } 
    (*F#
    with   
      member Name: string;

      /// This is ID of the algorithm used for the hashes of auxiliary 
      /// files in the assembly.   These hashes are stored in the 
      /// modulRefHash fields in this assembly. These are not cryptographic 
      /// hashes: they are simple file hashes. The algorithm is normally 
      /// 0x00008004 indicating the SHA1 hash algorithm.  
      member AuxModuleHashAlgorithm: i32; 

      member SecurityDecls: security_decls;

      /// This is the public key used to sign this 
      /// assembly (the signature itself is stored elsewhere: see the 
      /// binary format, and may not have been written if delay signing 
      /// is used).  (member Name, member PublicKey) forms the full 
      /// public name of the assembly.  
      member PublicKey: bytes option;  

      member Version: version_info option;
      member Locale: locale option;
      member CustomAttrs: custom_attrs;
      member Longevity: longevity; 
      member DisableJitOptimizations: bool;
      member JitTracking: bool;

      /// Records the types impemented by this asssembly in auxiliary 
      /// modules. 
      member ExportedTypes: exported_types;

      /// Records whether the entrypoint resides in another module. 
      member EntrypointElsewhere: modul_ref option;
    end
    F#*)
    
(* -------------------------------------------------------------------- 
 * One module in the "current" assembly, either a main-module or
 * an auxiliary module.  The main module will have a manifest.
 *
 * The abbreviation "modul" is used frequently throught the OCaml source
 * code because "module" is a resesrved word in OCaml.
 * -------------------------------------------------------------------- *)

type modul = (*F# ModuleDef and ModuleDef = F#*)   
    { modulManifest: manifest option;
      modulCustomAttrs: custom_attrs;
      modulName: string;
      modulTypeDefs: types;
      modulSubSystem: i32;
      modulDLL: bool;
      modulILonly: bool;
      modul32bit: bool;
      modulVirtAlignment: i32;
      modulPhysAlignment: i32;
      modulImageBase: i32;
      modulResources: resources; 
      modulNativeResources: bytes Lazy.t option; (* e.g. win86 resources, as the exact contents of a .res or .obj file *)
(*     modulFixups: fixups; *) }
(*F# with 
      member Manifest: manifest option;
      member CustomAttrs: custom_attrs;
      member Name: string;
      member TypeDefs: types;
      member SubSystemFlags: i32;
      member IsDLL: bool;
      member IsILOnly: bool;
      member Is32Bit: bool;
      member VirtualAlignment: i32;
      member PhysicalAlignment: i32;
      member ImageBase: i32;
      member Resources: resources; 
      member NativeResources: bytes Lazy.t option
     end F#*)

val manifest_of_mainmod: modul -> manifest
val module_is_mainmod: modul -> bool
val assname_of_mainmod: modul -> assembly_name

(* ====================================================================
 * PART 2
 * 
 * Making metadata.  Where no explicit constructor
 * is given, you should create the concrete datatype directly, 
 * e.g. by filling in all appropriate record fields.
 * ==================================================================== *)

(** These represent references to items in mscorlib. Version-neutral references can be generated using ecma_mscorlib_refs.  If you have already loaded a particular version of mscorlib you should reference items via an mscorlib_refs for that particular version of mscorlib built using mk_MscorlibRefs. *)
type mscorlib_refs = (*F# MscorlibRefs 
and 
 [<StructuralEquality(false); StructuralComparison(false)>]
 MscorlibRefs = F#*)
  { mscorlib_scoref: scope_ref;
    mscorlib_assembly_name: string;
    tref_Object: type_ref
    ; tspec_Object: type_spec
    ; typ_Object: typ
    ; tref_String: type_ref
    ; typ_String: typ
    ; typ_AsyncCallback: typ
    ; typ_IAsyncResult: typ
    ; typ_IComparable: typ
    ; tref_Type: type_ref
    ; typ_Type: typ
    ; tref_Missing: type_ref
    ; typ_Missing: typ
    ; typ_Activator: typ
    ; typ_Delegate: typ
    ; typ_ValueType: typ
    ; typ_Enum: typ
    ; tspec_TypedReference: type_spec
    ; typ_TypedReference: typ
    ; typ_MulticastDelegate: typ
    ; typ_Array: typ
    ; tspec_Int64: type_spec
    ; tspec_UInt64: type_spec
    ; tspec_Int32: type_spec
    ; tspec_UInt32: type_spec
    ; tspec_Int16: type_spec
    ; tspec_UInt16: type_spec
    ; tspec_SByte: type_spec
    ; tspec_Byte: type_spec
    ; tspec_Single: type_spec
    ; tspec_Double: type_spec
    ; tspec_IntPtr: type_spec
    ; tspec_UIntPtr: type_spec
    ; tspec_Char: type_spec
    ; tspec_Bool: type_spec
    ; typ_int8: typ
    ; typ_int16: typ
    ; typ_int32: typ
    ; typ_int64: typ
    ; typ_uint8: typ
    ; typ_uint16: typ
    ; typ_uint32: typ
    ; typ_uint64: typ
    ; typ_float32: typ
    ; typ_float64: typ
    ; typ_bool: typ
    ; typ_char: typ
    ; typ_int: typ
    ; typ_uint: typ
    ; typ_RuntimeArgumentHandle: typ
    ; typ_RuntimeTypeHandle: typ
    ; typ_RuntimeMethodHandle: typ
    ; typ_RuntimeFieldHandle: typ

    ; typ_Byte: typ
    ; typ_Int16: typ
    ; typ_Int32: typ
    ; typ_Int64: typ
    ; typ_SByte: typ
    ; typ_UInt16: typ
    ; typ_UInt32: typ
    ; typ_UInt64: typ
    ; typ_Single: typ
    ; typ_Double: typ
    ; typ_Bool: typ
    ; typ_Char: typ
    ; typ_IntPtr: typ
    ; typ_UIntPtr: typ
    ; tspec_Exception: type_spec
    ; typ_Exception: typ }

(** Build the table of commonly used references given a ScopeRef for mscorlib. *)
val mk_MscorlibRefs : scope_ref -> string option -> mscorlib_refs


(*F#
/// When writing a binary the fake "toplevel" type definition (called <Module>)
/// must come first. [dest_tdefs_with_toplevel_first] puts it first, and 
/// creates it in the returned list as an empty typedef if it 
/// doesn't already exist.
F#*)
val dest_tdefs_with_toplevel_first: mscorlib_refs -> types -> type_def list

(*F#
/// Note: not all custom attribute data can be decoded without binding types.  In particular 
/// enums must be bound in order to discover the size of the underlying integer. 
/// The following assumes enums have size int32. 
/// It also does not completely decode System.Type attributes 
F#*)
val decode_cattr_data: 
    mscorlib_refs -> 
    custom_attr -> 
      custom_attr_elem list *  (* fixed args *)
      custom_attr_named_arg list (* named args: values and flags indicating if they are fields or properties *) 

(* -------------------------------------------------------------------- 
 * Generate references to assemblies and modules
 * -------------------------------------------------------------------- *)

val mk_simple_assref: assembly_name -> assembly_ref
val mk_simple_modref: module_name -> modul_ref

(* -------------------------------------------------------------------- 
 * Make type refs
 * -------------------------------------------------------------------- *)

val mk_empty_gactuals: genactuals
val mk_tyvar_ty: Nums.u16 -> typ
val mk_genactual: typ -> genactual

val mk_nested_tref: scope_ref * string list * string -> type_ref
val mk_tref: scope_ref * string -> type_ref
val mk_tref_in_tref: type_ref * string -> type_ref

(* -------------------------------------------------------------------- 
 * Make type specs
 * -------------------------------------------------------------------- *)

val mk_nongeneric_tspec: type_ref -> type_spec
val mk_tspec: type_ref * genactuals -> type_spec
val mk_callconv: hasthis -> callconv

(* -------------------------------------------------------------------- 
 * Make types
 * -------------------------------------------------------------------- *)

val mk_typ: boxity -> type_spec -> typ
val mk_named_typ: boxity -> type_ref -> genactuals -> typ
val mk_boxed_typ: type_ref -> genactuals -> typ
val mk_value_typ: type_ref -> genactuals -> typ
val mk_nongeneric_boxed_typ: type_ref -> typ
val mk_nongeneric_value_typ: type_ref -> typ
val mk_array_ty: typ * array_shape -> typ
val mk_sdarray_ty: typ -> typ

(* -------------------------------------------------------------------- 
 * Make method references and specs
 * -------------------------------------------------------------------- *)

(* Construct references to any kind of method *)
val mk_mref: type_ref * callconv * string * int * typ list * typ -> method_ref
val mk_mspec: method_ref * boxity * genactuals * genactuals -> method_spec
val mk_mref_mspec_in_typ: method_ref * typ * genactuals -> method_spec
val mk_mspec_in_typ: typ * callconv * string * typ list * typ * genactuals -> method_spec

(* Construct references to methods on a given type *)
val mk_nongeneric_mspec_in_typ: typ * callconv * string * typ list * typ -> method_spec

(* Construct references to methods given a type_spec *)
val mk_mspec_in_tspec: type_spec * boxity * callconv * string * typ list * typ * genactuals -> method_spec
val mk_nongeneric_mspec_in_tspec: type_spec * boxity * callconv * string * typ list * typ -> method_spec

(* Construct references to instance methods *)
val mk_instance_mspec_in_tref: type_ref * boxity * string * typ list * typ * genactuals * genactuals -> method_spec
val mk_instance_mspec_in_tspec: type_spec * boxity * string * typ list * typ * genactuals -> method_spec
val mk_instance_mspec_in_typ: typ * string * typ list * typ * genactuals -> method_spec
val mk_instance_mspec_in_boxed_tspec: type_spec * string * typ list * typ * genactuals -> method_spec
val mk_instance_mspec_in_nongeneric_boxed_tref: type_ref * string * typ list * typ * genactuals -> method_spec

(* Construct references to non-generic methods *)
val mk_nongeneric_mspec_in_tref: type_ref * boxity * callconv * string * typ list * typ * genactuals -> method_spec
val mk_nongeneric_mspec_in_nongeneric_tref: type_ref * boxity * callconv * string * typ list * typ -> method_spec

(* Construct references to non-generic instance methods *)
val mk_nongeneric_instance_mspec_in_tref: type_ref * boxity * string * typ list * typ * genactuals -> method_spec
val mk_nongeneric_instance_mspec_in_tspec: type_spec * boxity * string * typ list * typ -> method_spec
val mk_nongeneric_instance_mspec_in_typ: typ * string * typ list * typ -> method_spec
val mk_nongeneric_instance_mspec_in_boxed_tspec: type_spec * string * typ list * typ -> method_spec
val mk_nongeneric_instance_mspec_in_nongeneric_boxed_tref: type_ref * string * typ list * typ -> method_spec

(* Construct references to static methods *)
val mk_static_mspec_in_nongeneric_boxed_tref: type_ref * string * typ list * typ * genactuals -> method_spec
val mk_static_mspec_in_boxed_tspec: type_spec * string * typ list * typ * genactuals -> method_spec
val mk_static_mspec_in_typ: typ * string * typ list * typ * genactuals -> method_spec

(* Construct references to static, non-generic methods *)
val mk_static_nongeneric_mspec_in_nongeneric_boxed_tref: type_ref * string * typ list * typ -> method_spec
val mk_static_nongeneric_mspec_in_boxed_tspec: type_spec * string * typ list * typ -> method_spec
val mk_static_nongeneric_mspec_in_typ: typ * string * typ list * typ -> method_spec

(* Construct references to toplevel methods in modules.  Usually compiler generated. *)
val mk_toplevel_static_mspec: scope_ref -> string * typ list * typ * genactuals -> method_spec
val mk_toplevel_static_nongeneric_mspec: scope_ref -> string * typ list * typ -> method_spec

(* Construct references to constructors *)
val mk_ctor_mspec: type_ref * boxity * typ list * genactuals -> method_spec
val mk_nongeneric_ctor_mspec: type_ref * boxity * typ list -> method_spec
val mk_ctor_mspec_for_boxed_tspec: type_spec * typ list -> method_spec
val mk_ctor_mspec_for_typ: typ * typ list -> method_spec
val mk_ctor_mspec_for_nongeneric_boxed_tref: type_ref * typ list -> method_spec

(* Construct references to fields *)
val mk_fref_in_tref: type_ref * string * typ -> field_ref
val mk_fspec: field_ref * typ -> field_spec
val mk_fspec_in_typ: typ * string * typ -> field_spec
val mk_fspec_in_tspec: type_spec * boxity * string * typ -> field_spec
val mk_fspec_in_boxed_tspec: type_spec * string * typ -> field_spec
val mk_fspec_in_nongeneric_boxed_tref: type_ref * string * typ -> field_spec

val mk_callsig: callconv * typ list * typ -> callsig


(* -------------------------------------------------------------------- 
 * Delegates
 * -------------------------------------------------------------------- *)
 
(*F#
/// Delegates.  Derived functions for telling if a method/class definition 
/// is really a delegate.  Also for telling if method signatures refer to 
/// delegate methods.
F#*)

(* these are approximations: you have to check that the type def is actually a delegate as well *)
val is_delegate_ctor: mscorlib_refs -> method_def -> bool
val is_delegate_invoke: mscorlib_refs -> method_def -> bool
val dest_delegate_invoke: mscorlib_refs -> method_def -> param list * returnv
val is_delegate_begin_invoke: mscorlib_refs -> method_def -> bool
val is_delegate_end_invoke: mscorlib_refs -> method_def -> bool
val dest_delegate_begin_end_invoke: method_def -> method_def -> param list * returnv


(* -------------------------------------------------------------------- 
 * Make generalized verions of possibly-generic types,
 * e.g. Given the type_def for List, return the type "List<T>".
 * -------------------------------------------------------------------- *)

val generalize_tref: type_ref -> genparam list -> type_spec
val generalize_tdef: scope_ref -> type_def -> type_spec
val generalize_nested_tdef: scope_ref -> type_def list * type_def -> type_spec

val gparams_add: genparams -> genparams -> genparams
val gparam_of_gactual: genactual -> genparam
val gparams_of_inst: genactuals -> genparams
val generalize_gparams: genparams -> genactuals

(* -------------------------------------------------------------------- 
 * Custom attributes 
 * -------------------------------------------------------------------- *)

val mk_custom_attribute_mref: 
    mscorlib_refs 
    -> method_spec 
       * custom_attr_elem list (* fixed args: values and implicit types *) 
       * custom_attr_named_arg list (* named args: values and flags indicating if they are fields or properties *) 
      -> custom_attr

val mk_custom_attribute: 
    mscorlib_refs 
    -> type_ref * typ list * 
       custom_attr_elem list (* fixed args: values and implicit types *) * 
       custom_attr_named_arg list (* named args: values and flags indicating if they are fields or properties *) 
       -> custom_attr

(* -------------------------------------------------------------------- 
 * Making code.
 * -------------------------------------------------------------------- *)

val check_code:  code -> code
val generate_code_label: unit -> code_label
val string_of_code_label : code_label -> string

(* Make some code that is a straight line sequence of instructions. *)
(* The function will add a "return" if the last instruction is not an exiting instruction *)
val nonbranching_instrs_to_code: instr list -> code 

(* Make some code that is a straight line sequence of instructions, then do *)
(* some control flow.  The first code label is the entry label of the generated code. *)
val nonbranching_instrs_then: code_label -> instr list -> instr -> code 
val nonbranching_instrs_then_ret: code_label -> instr list -> code
val nonbranching_instrs_then_br: code_label -> instr list -> code_label -> code

(* Make a basic block. The final instruction must be control flow *)
val nonbranching_instrs: code_label -> instr list -> code

(* For "join_code c1 c2", the unique exit of c1 should be the entry of c2, and the *)
(* label is then hidden in the result. *)
val join_code: code -> code -> code

(* Some more primitive helpers *)
val mk_bblock: basic_block -> code
val mk_scope_block: local_debug_info list * code -> code
val mk_group_block: code_label list * code list -> code
val mk_try_finally_block: code * code_label * code -> code
val mk_try_fault_block: code * code_label * code -> code

(* -------------------------------------------------------------------- 
 * Helpers for codegen: scopes for allocating new temporary variables.
 * -------------------------------------------------------------------- *)

type tmps 
val alloc_tmp: tmps -> local -> Nums.u16
val new_tmps : int -> tmps
val get_tmps : tmps -> local list

(* -------------------------------------------------------------------- 
 * Derived functions for making some common patterns of instructions
 * -------------------------------------------------------------------- *)

val mk_normal_call: method_spec -> instr
val mk_normal_callvirt: method_spec -> instr
val mk_normal_callconstraint: typ * method_spec -> instr
val mk_normal_newobj: method_spec -> instr
val mk_nongeneric_call_superclass_constructor: typ list * type_ref -> instr list
val mk_call_superclass_constructor: typ list * type_spec -> instr list
val mk_normal_stfld: field_spec -> instr
val mk_normal_stsfld: field_spec -> instr
val mk_normal_ldsfld: field_spec -> instr
val mk_normal_ldfld: field_spec -> instr
val mk_normal_ldflda: field_spec -> instr
val mk_normal_stind: basic_type -> instr
val mk_normal_ldind: basic_type -> instr
val mk_normal_cpind: basic_type -> instr list
val mk_normal_ldobj: typ -> instr
val mk_normal_stobj: typ -> instr 
val mk_ldc_i32: i32 -> instr
val ldarg_0: instr


val mk_tail_callvirt: method_spec -> instr  (* short term inclusion *)

val and_tailness: tailness -> bool -> tailness

(* -------------------------------------------------------------------- 
 * Derived functions for making return, parameter and local variable
 * objects for use in method definitions.
 * -------------------------------------------------------------------- *)

val mk_param: string option * typ -> param
val mk_unnamed_param: typ -> param
val mk_named_param: string * typ -> param
val mk_return: typ -> returnv
val mk_void_return: returnv
val mk_local: typ -> local

(* -------------------------------------------------------------------- 
 * Make a formal generic parameters
 * -------------------------------------------------------------------- *)

val mk_empty_gparams: genparams
val mk_simple_gparam: string -> genparam

(* -------------------------------------------------------------------- 
 * Make method definitions
 * -------------------------------------------------------------------- *)

val mk_ilmbody: bool (* initlocals? *) * local list * int * code * source option -> il_method_body
val mk_impl: bool * local list * int * code * source option -> method_body_details

val mk_ctor: member_access * param list * method_body_details -> method_def
val mk_nongeneric_nothing_ctor: source option -> type_ref -> param list -> method_def
val mk_nothing_ctor: source option -> type_spec -> param list -> method_def
val mk_static_mdef: genparams * string * member_access * param list * returnv * method_body_details -> method_def
val mk_static_nongeneric_mdef: string * member_access * param list * returnv * method_body_details -> method_def
val mk_cctor: method_body_details -> method_def
val mk_generic_virtual_mdef: string * member_access * genparams * (typ * typ list * typ) option * param list * returnv * method_body_details -> method_def
val mk_generic_instance_mdef: string * member_access * genparams * param list * returnv * method_body_details -> method_def
val mk_virtual_mdef: string * member_access * (typ * typ list * typ) option * param list * returnv * method_body_details -> method_def
val mk_instance_mdef: string * member_access * param list * returnv * method_body_details -> method_def
val mk_normal_virtual_mdef: string * member_access * param list * returnv  * method_body_details -> method_def

(* -------------------------------------------------------------------- 
 * Make field definitions
 * -------------------------------------------------------------------- *)

val mk_fdef: bool (* static? *) * string * typ * field_init option *  bytes option * member_access -> field_def
val mk_instance_fdef: string * typ * field_init option * member_access -> field_def
val mk_static_fdef: string * typ * field_init option * bytes option * member_access -> field_def

(* -------------------------------------------------------------------- 
 * Make a type definition
 * -------------------------------------------------------------------- *)

val add_mdef_to_tdef: method_def -> type_def -> type_def
val mk_generic_class: string * type_access * genparams * typ * typ list * methods * fields * properties * events * custom_attrs -> type_def
val mk_simple_tdef: mscorlib_refs -> string * type_access * methods * fields * properties * events * custom_attrs -> type_def
val mk_toplevel_tdef: mscorlib_refs -> methods * fields -> type_def
val add_fdef_to_tdef: field_def -> type_def -> type_def

(* -------------------------------------------------------------------- 
 * Make a type definition for a value type used to point to raw data.
 * These are useful when generating array initialization code 
 * according to the 
 *   ldtoken    field valuetype '<PrivateImplementationDetails>'/'$$struct0x6000127-1' '<PrivateImplementationDetails>'::'$$method0x6000127-1'
 *   call       void System.Runtime.CompilerServices.RuntimeHelpers::InitializeArray(class System.Array,valuetype System.RuntimeFieldHandle)
 * idiom.
 * -------------------------------------------------------------------- *)

val mk_rawdata_vtdef:  mscorlib_refs -> string * i32 (* size *) * u16 (* pack *) -> type_def

(* -------------------------------------------------------------------- 
 * Injecting code into existing code blocks.  A branch will
 * be added from the given instructions to the (unique) entry of
 * the code, and the first instruction will be the new entry
 * of the method.  The instructions should be non-branching.
 *
 * If you need to inject more complicated code sequences at the
 * start of a method, you can use "join_code" on the underlying "code"
 * for the method.
 * -------------------------------------------------------------------- *)

val prepend_instrs_to_code: instr list -> code -> code
val prepend_instrs_to_ilmbody: instr list -> il_method_body -> il_method_body
val prepend_instrs_to_mdef: instr list -> method_def -> method_def

(* -------------------------------------------------------------------- 
 * Injecting initialization code into a class.
 * Add some code to the end of the .cctor for a type.  Create a .cctor
 * if one doesn't exist already.
 * -------------------------------------------------------------------- *)

val prepend_instrs_to_cctor: instr list -> source option -> type_def -> type_def

(* -------------------------------------------------------------------- 
 * Derived functions for making some simple constructors
 * -------------------------------------------------------------------- *)

val mk_storage_ctor: source option -> instr list -> type_spec -> (string * typ) list -> method_def
val mk_simple_storage_ctor: source option -> type_spec option -> type_spec -> (string * typ) list -> method_def

val mk_delegate_mdefs: mscorlib_refs -> param list * returnv -> method_def list

(* -------------------------------------------------------------------- 
 * Given a delegate type definition which lies in a particular scope, 
 * make a reference to its constructor
 * -------------------------------------------------------------------- *)

val mk_ctor_mspec_for_delegate: mscorlib_refs -> type_ref * genactuals * bool -> method_spec 

(* -------------------------------------------------------------------- 
 * The toplevel "class" for a module or assembly.
 * -------------------------------------------------------------------- *)

val tref_for_toplevel: scope_ref -> type_ref
val tspec_for_toplevel: scope_ref -> type_spec
val typ_for_toplevel: scope_ref -> typ

(* -------------------------------------------------------------------- 
 * Making tables of custom attributes, etc.
 * -------------------------------------------------------------------- *)

val mk_custom_attrs: custom_attr list -> custom_attrs
val mk_computed_custom_attrs: (unit -> custom_attr list) -> custom_attrs
val add_custom_attr: custom_attr -> custom_attrs -> custom_attrs

val mk_security_decls: permission list -> security_decls
val mk_lazy_security_decls: (permission list) Lazy.t -> security_decls
val add_security_decl: permission -> security_decls -> security_decls

val mk_mbody : method_body_details -> method_body
val mk_lazy_mbody : method_body_details Lazy.t -> method_body

val mk_events: event_def list -> events
val mk_lazy_events: (event_def list) Lazy.t -> events
val add_event: event_def -> events -> events

val mk_properties: property_def list -> properties
val mk_lazy_properties: (property_def list) Lazy.t -> properties
val add_property: property_def -> properties -> properties

val mk_mdefs: method_def list -> methods
val mk_lazy_mdefs: (method_def list) Lazy.t -> methods
val add_mdef:  method_def -> methods -> methods

val mk_fdefs: field_def list -> fields
val mk_lazy_fdefs: (field_def list) Lazy.t -> fields
val add_fdef: field_def -> fields -> fields

val mk_mimpls: method_impl list -> method_impls
val mk_lazy_mimpls: (method_impl list) Lazy.t -> method_impls
val add_mimpl: method_impl -> method_impls -> method_impls

val mk_tdefs: type_def  list -> types

(*F#
/// Create table of types which is loaded/computed on-demand, and whose individual 
/// elements are also loaded/computed on-demand. Any call to [dest_tdefs] will 
/// result in the laziness being forced.  Operations can examine the
/// custom attributes and name of each type in order to decide whether
/// to proceed with examining the other details of the type.
/// 
/// Note that individual type definitions may contain further delays 
/// in their method, field and other tables. 
F#*)
val mk_lazy_tdefs: ((string list * string * custom_attrs * type_def Lazy.t) list) Lazy.t -> types
val add_tdef: type_def -> types -> types

val add_nested_exported_type: nested_exported_type -> nested_exported_types -> nested_exported_types
val mk_nested_exported_types: nested_exported_type list -> nested_exported_types
val mk_lazy_nested_exported_types: (nested_exported_type list) Lazy.t -> nested_exported_types

val mk_exported_types: exported_type list -> exported_types
val mk_lazy_exported_types: (exported_type list) Lazy.t -> exported_types
val add_exported_type: exported_type -> exported_types -> exported_types

val mk_resources: resource list -> resources
val add_resource: resource -> resources -> resources
val mk_lazy_resources: (resource list) Lazy.t -> resources

(* -------------------------------------------------------------------- 
 * Errors arising from building opaque tables of methods, types, fields
 * etc.
 * -------------------------------------------------------------------- *)

exception Not_unique_field of field_def
exception Not_unique_method of method_def
exception Not_unique_type of string

(* -------------------------------------------------------------------- 
 * Making modules
 * -------------------------------------------------------------------- *)

val mk_simple_mainmod: assembly_name -> module_name -> bool (* dll? *) -> types -> modul
val add_toplevel_mdef: mscorlib_refs -> method_def -> modul -> modul

(* -------------------------------------------------------------------- 
 * Default values for some of the strange flags in a module.
 * -------------------------------------------------------------------- *)

val default_modulSubSystem: Nums.i32
val default_modulVirtAlignment: Nums.i32
val default_modulPhysAlignment: Nums.i32
val default_modulImageBase: Nums.i32

(* -------------------------------------------------------------------- 
 * Generate references to existing type definitions, method definitions
 * etc.  Useful for generating references, e.g. to a  class we're processing
 * Also used to reference type definitions that we've generated.  [scope_ref] 
 * is normally ScopeRef_local, unless we've generated the type_def in
 * an auxiliary module or are generating multiple assemblies at 
 * once.
 * -------------------------------------------------------------------- *)

val tref_for_tdef: scope_ref -> type_def -> type_ref

val tref_for_nested_tdef : scope_ref -> type_def list * type_def               -> type_ref
val tspec_for_nested_tdef: scope_ref -> type_def list * type_def               -> type_spec
val mref_for_mdef        : scope_ref -> type_def list * type_def -> method_def -> method_ref
val fref_for_fdef        : scope_ref -> type_def list * type_def -> field_def  -> field_ref

val mk_mspec_to_mdef: typ * method_def * genactuals -> method_spec
val mk_mref_to_mdef: type_ref * method_def -> method_ref
val mk_fref_to_fdef: type_ref * field_def -> field_ref

val assref_for_manifest: manifest -> assembly_ref
val assref_for_mainmod: modul -> assembly_ref
val modref_for_modul: modul -> modul_ref


(* ====================================================================
 * PART 3: Utilities
 * ==================================================================== *)

(* -------------------------------------------------------------------- 
 * Rescoping.
 *
 * Given an object O1 referenced from where1 (e.g. O1 binds to some  
 * result R when referenced from where1), and given that SR2 resolves to where1 from where2, 
 * produce a new O2 for use from where2 (e.g. O2 binds to R from where2)
 *
 * So, scope_ref tells you how to reference the original scope from 
 * the new scope. e.g. if scope_ref is:
 *    [ScopeRef_local] then the object is returned unchanged
 *    [ScopeRef_module m] then an object is returned 
 *                        where all ScopeRef_local references 
 *                        become ScopeRef_module m
 *    [ScopeRef_assembly m] then an object is returned 
 *                        where all ScopeRef_local and ScopeRef_module references 
 *                        become ScopeRef_assembly m
 * -------------------------------------------------------------------- *)

(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_scoref: scope_ref -> scope_ref -> scope_ref
(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_tspec: scope_ref -> type_spec -> type_spec
(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_typ: scope_ref -> typ -> typ
(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_mspec: scope_ref -> method_spec -> method_spec
(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_ospec: scope_ref -> overrides_spec -> overrides_spec
(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_mref: scope_ref -> method_ref -> method_ref 
(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_fref: scope_ref -> field_ref -> field_ref
(*F# 
/// Rescoping. The first argument tells the function how to reference the original scope from 
/// the new scope. 
F#*)
val rescope_fspec: scope_ref -> field_spec -> field_spec



(*-----------------------------------------------------------------------
 * The Code Builder utility.
 *----------------------------------------------------------------------*)

type seh_clause = 
  | SEH_finally of (code_label * code_label)
  | SEH_fault  of (code_label * code_label)
  | SEH_filter_catch of (code_label * code_label) * (code_label * code_label)
  | SEH_type_catch of typ * (code_label * code_label)

type exception_spec = 
    { exnRange: (code_label * code_label);
      exnClauses: seh_clause list }

type local_spec = 
    { locRange: (code_label * code_label);
      locInfos: local_debug_info list }

(*F#
/// build_code: Build code from a sequence of instructions.
/// 
/// e.g. "build_code meth resolver instrs exns locals"
/// 
/// This makes the basic block structure of code from more primitive
/// information, i.e. an array of instructions.
///   [meth]: for debugging and should give the name of the method.
///   [resolver]: should return the instruction indexes referred to 
///               by code-label strings in the instruction stream.
///   [instrs]: the instructions themselves, perhaps with attributes giving 
///             debugging information
///   [exns]: the table of exception-handling specifications
///           for the method.  These are again given with respect to labels which will
///           be mapped to pc's by [resolver].  
///   [locals]: the table of specifications of when local variables are live and
///           should appear in the debug info.
/// 
/// If the input code is well-formed, the function will returns the 
/// chop up the instruction sequence into basic blocks as required for
/// the exception handlers and then return the tree-structured code
/// corresponding to the instruction stream.
/// A new set of code labels will be used throughout the resulting code.
/// 
/// The input can be badly formed in many ways: exception handlers might
/// overlap, or scopes of local variables may overlap badly with 
/// exception handlers.
F#*)
val build_code:
    string ->
    (code_label -> int) -> 
    instr array -> 
    exception_spec list -> 
    local_spec list -> 
    code

(* -------------------------------------------------------------------- 
 * The instantiation utilities.
 * -------------------------------------------------------------------- *)

(** Instantiate type variables that occur within types and other items. *)
val inst_tspec_aux: int -> genactuals -> type_spec -> type_spec

(** Instantiate type variables that occur within types and other items. *)
val inst_typ_aux: int -> genactuals -> typ -> typ

(** Instantiate type variables that occur within types and other items. *)
val inst_genactual_aux: int -> genactuals -> genactual -> genactual

(** Instantiate type variables that occur within types and other items. *)
val inst_inst_aux: int -> genactuals -> genactuals -> genactuals

(** Instantiate type variables that occur within types and other items. *)
val inst_callsig_aux: int -> genactuals -> callsig -> callsig

(** Instantiate type variables that occur within types and other items. *)
val inst_typ: genactuals -> typ -> typ

(** Instantiate type variables that occur within types and other items. *)
val inst_inst: genactuals -> genactuals -> genactuals

(** Instantiate type variables that occur within types and other items. *)
val inst_tspec: genactuals -> type_spec -> type_spec

(** Instantiate type variables that occur within types and other items. *)
val inst_callsig: genactuals -> callsig -> callsig

(** Instantiate type variables that occur within types and other items. *)
val inst_read: genactuals -> Nums.u16 -> genactual

(** Instantiate type variables that occur within types and other items. *)
val inst_add: genactuals -> genactuals -> genactuals

(* -------------------------------------------------------------------- 
 * Names of some commonly used things in mscorlib...
 * -------------------------------------------------------------------- *)

(*val mscorlib_assembly_name: string --- removed *)
val mscorlib_module_name: string


(** This is a 'vendor neutral' way of referencing mscorlib. *)
val ecma_mscorlib_assembly_name:string
(** This is a 'vendor neutral' way of referencing mscorlib. *)
val ecma_public_token: public_key_info
(** This is a 'vendor neutral' way of referencing mscorlib. *)
val ecma_mscorlib_assref: assembly_ref
(** This is a 'vendor neutral' way of referencing mscorlib. *)
val ecma_mscorlib_scoref: scope_ref
(** This is a 'vendor neutral' collection of references to items in mscorlib. *)
val ecma_mscorlib_refs: mscorlib_refs
(** This is a 'vendor neutral' collection of references to items in mscorlib. Equivalent to ecma_mscorlib_refs *)
(*val mscorlib : mscorlib_refs --- removed *)


(* Some commonly used methods *)
val mspec_RuntimeHelpers_InitializeArray: mscorlib_refs -> method_spec 
val mspec_RunClassConstructor: mscorlib_refs -> method_spec
val mk_RunClassConstructor: mscorlib_refs -> type_spec -> instr list

val mk_writeline_call: mscorlib_refs -> instr
val mk_mscorlib_exn_newobj: mscorlib_refs -> string -> instr

(* Some commonly used custom attibutes *)
val mk_DebuggableAttribute: mscorlib_refs -> bool (* debug tracking *) * bool (* disable JIT optimizations *) -> custom_attr
val mk_DebuggableAttribute_v2: mscorlib_refs -> bool (* jitTracking *) * bool (* ignoreSymbolStoreSequencePoints *) * bool (* disable JIT optimizations *) * bool (* enable EnC *) -> custom_attr
val mk_CompilerGeneratedAttribute: mscorlib_refs -> custom_attr

(* -------------------------------------------------------------------- 
 * Discriminating different important built-in types
 * -------------------------------------------------------------------- *)

(*val typ_is_MulticastDelegate: typ -> bool*)
(*val typ_is_Delegate: typ -> bool*)
(*val typ_is_ValueType: typ -> bool*)
(*val typ_is_Enum: typ -> bool*)
val typ_is_Object: mscorlib_refs -> typ -> bool
val typ_is_String: mscorlib_refs -> typ -> bool
val typ_is_SByte: mscorlib_refs -> typ -> bool
val typ_is_Byte: mscorlib_refs -> typ -> bool
val typ_is_Int16: mscorlib_refs -> typ -> bool
val typ_is_UInt16: mscorlib_refs -> typ -> bool
val typ_is_Int32: mscorlib_refs -> typ -> bool
val typ_is_UInt32: mscorlib_refs -> typ -> bool
val typ_is_Int64: mscorlib_refs -> typ -> bool
val typ_is_UInt64: mscorlib_refs -> typ -> bool
val typ_is_IntPtr: mscorlib_refs -> typ -> bool
val typ_is_UIntPtr: mscorlib_refs -> typ -> bool
val typ_is_Bool: mscorlib_refs -> typ -> bool
val typ_is_Char: mscorlib_refs -> typ -> bool
val typ_is_TypedReference: mscorlib_refs -> typ -> bool
val typ_is_Double: mscorlib_refs -> typ -> bool
val typ_is_Single: mscorlib_refs -> typ -> bool

(* -------------------------------------------------------------------- 
 * Get a public key token from a public key.
 * -------------------------------------------------------------------- *)

val sha1_hash_bytes : bytes -> bytes (* SHA1 hash *)
val token_from_public_key: bytes -> bytes (* SHA1 hash of public key chopped to last 8 bytes *)

(* -------------------------------------------------------------------- 
 * Get a version number from a CLR version string, e.g. 1.0.3705.0
 * -------------------------------------------------------------------- *)

val parse_version: string -> version_info
val version_to_string: version_info -> string
val version_compare: version_info -> version_info -> int
val version_max: version_info -> version_info -> version_info
val version_min: version_info -> version_info -> version_info

(* -------------------------------------------------------------------- 
  * Decompose a type definition according to its kind.
 * -------------------------------------------------------------------- *)

type enum_info (*F# = EnumInfo and EnumInfo F#*) =  
    { enumValues: (string * field_init) list;  
      enumType: typ }

val values_of_enum_info: enum_info -> (string * field_init) list
val typ_of_enum_info: enum_info -> typ

val info_for_enum: string * fields -> enum_info

(* -------------------------------------------------------------------- 
 * The resource and sharing management utilities.  Essential if you need
 * to maintain sharing while manipulating very large graphs.
 * 
 * Some intern tables.  Returned values are guaranteed to be 
 * memory-shared with other identical values. Terms associated with 
 * different managers may end up with identical indexes.  Identical terms tagged
 * using tag_XYZ do not get unique tags.
 * -------------------------------------------------------------------- *)

type idx = int
type 'a idx_interner = 'a -> 'a * idx

type manager (*F# = TermManager and TermManager F#*)

val new_manager: unit -> manager

val intern_bytes:  manager option -> bytes       -> bytes 
val intern_string: manager option -> string      -> string 
val intern_tref:   manager option -> type_ref    -> type_ref 
val intern_tspec:  manager option -> type_spec   -> type_spec 
val intern_mspec:  manager option -> method_spec -> method_spec 
val intern_typ:    manager option -> typ         -> typ 
val intern_instr:  manager option -> instr       -> instr 

val tag_bytes:  manager option -> bytes       -> bytes 
val tag_string: manager option -> string      -> string 
val tag_tref:   manager option -> type_ref    -> type_ref 
val tag_tspec:  manager option -> type_spec   -> type_spec 
val tag_mspec:  manager option -> method_spec -> method_spec 
val tag_typ:    manager option -> typ         -> typ 
val tag_instr:  manager option -> instr       -> instr 

val memoize_on: ('a -> 'key) -> ('a -> 'b) -> ('a -> 'b)
val memoize: ('a -> 'b) -> ('a -> 'b)
val mspec_idx:  manager -> method_spec -> idx

(* -------------------------------------------------------------------- 
 * For completeness.  These do not occur in metadata but tools that
 * care about the existence of properties and events in the metadata
 * can benefit from them.
 * -------------------------------------------------------------------- *)
 
type event_ref (*F# = EventRef 
and EventRef 
  with 
    static member Create : type_ref * string -> EventRef
    member EnclosingTypeRef: type_ref
    member Name: string
  end F#*)

type event_spec (*F# = EventSpec 
and EventSpec 
  with 
     static member Create : event_ref * typ -> EventSpec
     member EventRef: event_ref
     member EnclosingType: typ
  end F#*)

type property_ref (*F# = PropertyRef 
and PropertyRef 
  with 
    static member Create : type_ref * string -> PropertyRef
    member EnclosingTypeRef: type_ref
    member Name: string
  end  F#*)

type property_spec (*F# =  PropertySpec 
and PropertySpec 
  with 
    static member Create : property_ref * typ -> PropertySpec
    member PropertyRef: property_ref
    member EnclosingType: typ
  end F#*)

val tref_of_pref : property_ref -> type_ref
val tref_of_eref : event_ref -> type_ref
val mk_pref : type_ref * string -> property_ref
val mk_eref : type_ref * string -> event_ref
val mk_pspec : property_ref * typ -> property_spec
val mk_espec : event_ref * typ -> event_spec
val name_of_pref : property_ref -> string
val name_of_eref : event_ref -> string
val enclosing_typ_of_pspec : property_spec -> typ
val enclosing_typ_of_espec : event_spec -> typ
val pref_of_pspec : property_spec -> property_ref
val eref_of_espec : event_spec -> event_ref
val eref_for_edef        : scope_ref -> type_def list * type_def -> event_def  -> event_ref
val pref_for_pdef        : scope_ref -> type_def list * type_def -> property_def  -> property_ref


(* -------------------------------------------------------------------- 
 * The referenced-assemblies utility.
 *
 * -------------------------------------------------------------------- *)

type refs = (*F# References and References = F#*) 
    { refsAssembly: assembly_ref list; 
      refsModul: modul_ref list; }
    (*F# with 
            member AssemblyReferences: assembly_ref list
            member ModuleReferences: modul_ref list
         end
    F#*)

(** Find the full set of assemblies referenced by a module *)
val refs_of_module: modul -> refs
val empty_refs: refs

(* -------------------------------------------------------------------- 
 * The following functions are used to define an extension to the IL.
 *
 * Don't try this at home!  At the moment this is currently strictly 
 * for the use of MS-ILX only (i.e. ilx.ml), and only one such extension 
 * can be defined.  This may change in future releases.
 * -------------------------------------------------------------------- *)

type 'a instr_extension = 
    { instrExtDests: ('a -> code_label list);
      instrExtFallthrough: ('a -> code_label option);
      instrExtIsTailcall: ('a -> bool);
      instrExtRelabel: (code_label -> code_label) -> 'a -> 'a; }

type 'a typ_extension = 
    { typeExtRescope: (scope_ref -> 'a -> 'a);
      typeExtInstAux: (int -> genactuals -> 'a -> 'a) }

type 'a type_def_kind_extension = Type_def_kind_extension

val define_typ_extension: 'a typ_extension -> ('a -> ext_typ) * (ext_typ -> bool) * (ext_typ -> 'a)
val define_instr_extension: 'a instr_extension -> ('a -> ext_instr) * (ext_instr -> bool) * (ext_instr -> 'a)
val define_type_def_kind_extension: 'a type_def_kind_extension -> ('a -> ext_type_def_kind) * (ext_type_def_kind -> bool) * (ext_type_def_kind -> 'a)
