(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
module Microsoft.FSharp.Compiler.Tast 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 

module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Il = Microsoft.Research.AbstractIL.IL 
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Ccuthunk = Microsoft.FSharp.Compiler.CcuThunk 
F#*) 
open Ildiag
open List
open Range
open Ast
open Lib
open Printf
open Ccuthunk

(*-------------------------------------------------------------------------
!* uniq
 *------------------------------------------------------------------------*)

let new_uniq = let i = ref 0 in fun () -> incr i; !i
let new_stamp = let i = ref 0 in fun () -> incr i; !i

(*---------------------------------------------------------------------------
!* Signatures/Types
 *------------------------------------------------------------------------- *)

(* NOTE: Although tycons, exncs and moduls are all modelled via tycon_specs, *)
(* they have different name-resolution logic. *)
(* For example, an excon ABC really correspond to a type called *)
(* ABCException with a data constructor ABC. At the moment they are *)
(* simply indexed in the excon table as the discriminator constructor ABC. *)
(* Hence we currently have multiple tables for these entities. *)
(* REVIEW: long term aim is to combine these into a single table. *)
type tycon_spec = tycon_spec_data osgn
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tycon_spec_data =
    { (** The declared type parameters of the type  *)
      (* MUTABILITY; used only during creation and remapping  of tycons *)
      mutable tycon_typars: typars;        
      
      (** The unique stamp of the "tycon blob". Note the same tycon in signature and implementation get different stamps *)
      tycon_stamp: int;

      (** The name of the type, possibly with `n mangling *)
      tycon_name: string;

      (** The declaration location for the type constructor *)
      tycon_range: range;
      
      (** Indicates the type prefers the "tycon<a,b>" syntax for display etc. *)
      tycon_prefix_display: bool;                   
      
      (** Indicates the "tycon blob" is actually a module *)
      tycon_is_modul : bool; 

      (** The declared accessibility of the representation, not taking signatures into account *)
      tycon_repr_access: taccess;
      
      (** The declared attributes for the type *)
      (* MUTABILITY; used only during creation and remapping of tycons *)
      mutable tycon_attribs: attribs;     
                
      (** The declared representation of the type, i.e. record, union, class etc. *)
      mutable tycon_repr: tycon_repr option;   (* MUTABILITY; used only during creation and remapping of tycons *)

      (** If non-None, indicates the type is an abbreviation for another type. *)
      mutable tycon_abbrev: typ option;             (* MUTABILITY; used only during creation and remapping of tycons *)
      
      (** The methods and properties of the type *)
      mutable tycon_tcaug: tycon_augmentation;      (* MUTABILITY; used only during creation and remapping of tycons *)
      
      (** Field used when the 'tycon' is really an exception definition *)
      (* MUTABILITY; used only during creation and remapping of tycons *)
      mutable tycon_exnc_info: tycon_exnc_info;     
      
      (** This field is used when the 'tycon' is really a module definition. It holds statically nested type definitions and nested modules *)
      (* MUTABILITY: only used during creation and remapping  of tycons and *)
      (* when compiling fslib to fixup compiler forward references to internal items *)
      mutable tycon_modul_contents: modul_typ Lazy.t;     

      (** The declared documentation for the type or module *)
      tycon_xmldoc : xmlDoc;

      (** The stable path to the type, e.g. Microsoft.FSharp.Core.FastFunc`2 *)
      (* REVIEW: it looks like tycon_cpath subsumes this *)
      tycon_pubpath : public_path option; (*   where does this live? *)

      mutable tycon_access: taccess; (*   how visible is this? *)  (* MUTABILITY; used only during creation and remapping  of tycons *)
 
      (** The stable path to the type, e.g. Microsoft.FSharp.Core.FastFunc`2 *)
      tycon_cpath : compilation_path option; 

      (** Used during codegen to hold the ILX representation indicating how to access the type *)
      tycon_il_repr_cache : il_type_repr cache;  (* MUTABILITY; *)

    }

and parent_ref = 
    | Parent of tycon_ref
    | ParentNone
    
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tycon_augmentation = 
    { (* This is the value implementing the auto-generated comparison *)
      (* semantics if any. It is not present if the type defines its own implementation *)
      (* of IComparable or if the type doesn't implement IComparable implicitly. *)
      mutable tcaug_compare        : val_ref option;             

      (* This is the value implementing the auto-generated equality *)
      (* semantics if any. It is not present if the type defines its own implementation *)
      (* of Object.Equals or if the type doesn't override Object.Equals implicitly. *)
      mutable tcaug_equals        : (val_ref * val_ref) option;             

      (* True if the type defined an Object.GetHashCode method. In this case we give a warning if we auto-generate a hash method since the semantics may not match up *)
      mutable tcaug_hasObjectGetHashCode : bool;             
      
      (* Likewise IStructuralHash::GetHashCode *)
      mutable tcaug_structural_hash: val_ref option;             
      
      (* Properties, methods etc. *)
      mutable tcaug_adhoc          : (val_ref list) namemap;
      
      (* Interface implementations - boolean indicates compiler-generated *)
      mutable tcaug_implements     : (typ * bool * range) list;  
      
      (* Super type, if any *)
      mutable tcaug_super          : typ option;                 
      
      (* Set to true at the end of the scope where proper augmentations are allowed *)
      mutable tcaug_closed         : bool;                       

      (* Set to true if the type is determined to be abstract *)
      mutable tcaug_abstract : bool;                       
    }
   
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tycon_repr = 
    (** Indicates the type is a class, struct, enum, delegate or interface *)
    | TFsObjModelRepr    of tycon_objmodel_data
    (** Indicates the type is a record *)
    | TRecdRepr          of tycon_rfields
    (** Indicates the type is a discriminatd union *)
    | TFiniteUnionRepr   of tycon_funion_data 
    (** Indicates the type is a .NET type *)
    | TIlObjModelRepr    of 
          (* scope: *)      Il.scope_ref * 
          (* nesting: *)    Il.type_def list * 
          (* definition: *) Il.type_def 
    (** Indicates the type is implemented as IL assembly code using the given closed Abstract IL type *)
    | TAsmRepr           of Il.typ


and 
  tycon_objmodel_kind = 
    (** Indicates the type is a class *)
    | TTyconClass 
    (** Indicates the type is an interface *)
    | TTyconInterface 
    (** Indicates the type is a struct *)
    | TTyconStruct 
    (** Indicates the type is a delegate with the given Invoke signature *)
    | TTyconDelegate of tslotsig 
    (** Indicates the type is an enumeration *)
    | TTyconEnum
    
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tycon_objmodel_data = 
    { (** Indicates whether the type declaration is a class, interface, enum, delegate or struct *)
      tycon_objmodel_kind: tycon_objmodel_kind;
      (** The declared abstract slots of the class, interface or struct *)
      fsobjmodel_vslots: val_ref list; 
      (** The fields of the class, struct or enum *)
      fsobjmodel_rfields: tycon_rfields }

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tycon_rfields = 
    { (** The fields of the record, in declaration order. *)
      rfields_by_index: recdfield_spec array;
      
      (** The fields of the record, indexed by name. *)
      rfields_by_name : recdfield_spec namemap  }

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tycon_uconstrs_spec = 
    { (** The cases of the discriminated union, in declaration order. *)
      uconstrs_by_index: unionconstr_spec array;
      (** The cases of the discriminated union, indexed by name. *)
      uconstrs_by_name : unionconstr_spec namemap }

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tycon_funion_data =
    { (** The cases contained in the discriminated union. *)
      funion_constrs: tycon_uconstrs_spec;
      (** The ILX data structure representing the discriminated union. *)
      funion_ilx_repr: Ilx.classunion_ref cache; }

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  unionconstr_spec =
    { (** Data carried by the case. *)
      uconstr_rfields: tycon_rfields;
      (** Return type constructed by the case. Normally exactly the type of the enclosing type, sometimes an abbreviation of it *)
      uconstr_rty: typ;
      (** Name of the case in generated IL code *)
      uconstr_il_name: string;
      (** Documentation for the case *)
      uconstr_xmldoc : xmlDoc;
      (** Name/range of the case *)
      uconstr_id: Ast.ident; 
      (**  Indicates the declared visibility of the union constructor, not taking signatures into account *)
      uconstr_access: taccess; 
      (** Attributes, attached to the generated static method to make instances of the case *)
      uconstr_attribs: attribs; }

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  recdfield_spec =
    { (** Is the field declared mutable in F#? *)
      rfield_mutable: bool;
      (** Documentation for the field *)
      rfield_xmldoc : xmlDoc;
      (** The type of the field, w.r.t. the generic parameters of the enclosing type constructor *)
      rfield_type: typ;
      (** Indicates a static field *)
      rfield_static: bool;
      (** Indicates a compiler generated field, not visible to Intellisense or name resolution *)
      rfield_secret: bool;
      (** The default initialization info, for static literals *)
      rfield_const: tconst option; 
      (**  Indicates the declared visibility of the field, not taking signatures into account *)
      rfield_access: taccess; 
      (** Attributes attached to generated property *)
      rfield_pattribs: attribs; 
      (** Attributes attached to generated field *)
      rfield_fattribs: attribs; 
      (** Name/declaration-location of the field *)
      rfield_id: Ast.ident; }

and tycon_exnc_info =
    (** Indicates that an exception is an abbreviation for the given exception *)
    | TExnAbbrevRepr of tycon_ref 
    (** Indicates that an exception is shorthand for the given .NET exception type *)
    | TExnAsmRepr of Il.type_ref
    (** Indicates that an exception carries the given record of values *)
    | TExnFresh of tycon_rfields
    (** Indicates that an exception is abstract, i.e. is in a signature file, and we do not know the representation *)
    | TExnNone

and modul_kind = 
    (** Indicates that a module is compiled to a class with the given mangled name. The mangling is reversed during lookup *)
    | AsMangledNamedType of string 
    (** Indicates that a module is compiled to a class with the same name as the original module *)
    | AsNamedType 
    (** Indicates that a 'module' is really a namespace *)
    | Namespace

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  modul_typ = 
    { (** Namespace or module-compiled-as-type? *)
      mtyp_kind: modul_kind;  
              
      (** The nested modules inside this module, if any *)
      (* MUTATION: only used when compiling fslib to fixup compiler forward references to internal items *)
      mutable mtyp_submoduls: modul_spec namemap;  
      
      (** Values, including members in F# types in this module-or-namespace-fragment. *)
      (* TODO: why are we using a namemap here, or in any of these fields? *)
      (* This is a curse. Just use a list and build the *)
      (* name resolution environment from that list. Then we don't need internal names for  *)
      (* values. *)
      mtyp_vals             : val_spec   namemap; 
      
      (** Type, mapping mangled name to tycon_spec, e.g. "Dictionary`2" --> tycon_spec *)
      mtyp_tycons           : tycon_spec namemap; 

      (** Lookup tables keyed the way various clients expect them to be keyed *)
      (* We attach them here so we don't need to store lookup tables via any other technique *)
      (* TODO: remove the use of soft-typing here unless where strictly necessary *)
      mtyp_apref_cache                        : Obj.t (* apelem_ref namemmap *) option ref;
      mtyp_exconsByDemangledName_cache        : Obj.t (* tycon_ref namemmap *)  option ref;
      mtyp_tyconsByDemangledNameAndArity_cache: Obj.t (* tycon_ref namemmap *)  option ref;
      mtyp_tyconsByAccessNames_cache          : Obj.t (* tycon_ref namemmmap *) option ref; }
  
and modul_spec = tycon_spec 

and taccess = 
    (** TAccess(...,path,...) indicates the construct  can only be accessed from any code in the given type constructor, module or assembly. [] indicates global scope. *)
    | TAccess of compilation_path list
    
and typar_spec = typar_spec_data osgn
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  typar_spec_data = 
     { mutable typar_id: ident; (* MUTABILITY: we hack the names of generalized inference type parameters to make the look nice for IL code generation *)
       
       mutable typar_flags: int32;
       (*
          FLAGS ARE LOGICALLY: 
              (* MUTABILITY CLEANUP: could create fresh rigid variables and equate these to them. *) 
              mutable typar_rigid: bool;                                 (* cannot unify: quantified.  Mutated when inference decides to generalize. *)
              typar_from_error: bool;                                    (* typar was generated as part of error recovery *)
              typar_compgen: bool;
              mutable typar_static_req: typarStaticReq;                  (* true for $a types or any tyvars in types equated with $a types - these may not be generalized *)
       *)
       
       (** The unique stamp of the typar blob. *)
       typar_stamp: int; 
       
       (** The documentation for the type parameter. Empty for type inference variables.*)
       typar_xmldoc : xmlDoc;
       
       (** The decalred attributes of the type parameter. Empty for type inference variables. *)
       mutable typar_attribs: attribs;                      
       
       (** An inferred equivalence for a type inference variable. *)
       (* Note: this is the most important mutable state in all of F#! *)
       mutable typar_solution: typ;                      
       
       (** The inferred constraints for the type inference variable *)
       (* Note: along with typar_solution, this is the most important mutable state in all of F#! *)
       mutable typar_constraints: typar_constraint list; 
     } 

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  typar_constraint = 
    (** Indicates a constraint that a type is a subtype of the given type *)
    | TTyparCoercesToType              of typar_constraint_typ * range

    (** Indicates a default value for an inference type variable should it be netiher generalized nor solved *)
    | TTyparDefaultsToType             of int * typ * range 
    
    (** Indicates a constraint that a type has a 'null' value *)
    | TTyparSupportsNull               of range 
    
    (** Indicates a constraint that a type has a member with the given signature *)
    | TTyparMayResolveMemberConstraint of trait_constraint_info * range 
    
    (** Indicates a constraint that a type is a non-Nullable value type *)
    (* These are part of .NET's model of generic constraints, and in order to *)
    (* generate verifiable code we must attach them to F# generalzied type variables as well. *)
    | TTyparIsNotNullableValueType     of range 
    
    (** Indicates a constraint that a type is a reference type *)
    | TTyparIsReferenceType            of range 

    (** Indicates a constraint that a type is a simple choice between one of the given ground types. See format.ml *)
    | TTyparSimpleChoice               of typ list * range 

    (** Indicates a constraint that a type has a parameterless constructor *)
    | TTyparRequiresDefaultConstructor of range 

    (** Indicates a constraint that a type is an enum with the given underlying *)
    | TTyparIsEnum                     of typ * range 
    
    (** Indicates a constraint that a type is a delegate from the given tuple of args to the given return type *)
    | TTyparIsDelegate                 of typ * typ * range 
    
and trait_constraint_info = 
    (** Indicates the signature of a member constraint *)
    | TTrait of typ list * string * memberFlags * typ list * typ

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  typar_constraint_typ = 
    (* These are only attached to type paramters from IL Code.  These are always *)
    (* read and copied from the IL metadata and turned TTyparSubtypeConstraintFromFS *)
    (* before they are used.  This means we avoid some circularities in the process of *)
    (* reading the constraints from .NET metadata. *)
    | TTyparSubtypeConstraintFromIL of Il.scope_ref * Il.typ  
    (** These are the true representation of the constraints. *)
    | TTyparSubtypeConstraintFromFS of typ
  
and val_mutability   = 
    | Immutable 
    | Mutable

and val_base_or_this = 
    | CtorThisVal 
    | BaseVal 
    | NormalVal 
    | MemberThisVal

and val_spec = val_spec_data osgn

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  val_spec_data =
    { (** The name the value. *)
      val_name: string;
      
      (** The place where it was defined. *)
      val_range: range;

      (** Range of the definition (implementation) of the value, used by Visual Studio *)
      (* Updated by mutation when the implementation is matched against the signature. *)
      mutable val_defn_range: range; 
      
      (** The type of the value. *)
      (* May be a Type_forall for a generic value. *)
      (* May be a type variable or type containing type variables during type inference. *)

      (* Mutability used only in inference by adjustAllUsesOfRecValue.  *)
      (* This replaces a recursively inferred type with a schema. *)
      (* MUTABILITY CLEANUP: find a way to do this using type unification alone. *)
      mutable val_type: typ;
      
      (** A unique stamp within the context of this invocation of the compiler process *)
      val_stamp: int; 
      
      (** See vflags section further below for encoding/decodings here *)
      mutable val_flags: int64;

      (** The value of a value or member marked with [<LiteralAttribute>] *)
      mutable val_const: tconst option;

      (** What is the compilation path? *)
      val_pubpath : public_path option;

      (** What is the original, unoptimized, closed-term definition, if any? *)
      mutable val_defn: expr option; 

      (* How visible is this? *)
      val_access: taccess; 

      (* Is the value actually an instance method/property/event that augments *)
      (* a type, and if so what name does it take in the IL?*)
      val_meminfo: val_member_info;

      (* Custom attributes attached to the value. These contain references to other values (i.e. constructors in types). Mutable to fixup  *)
      (* these value references after copying a colelction of values. *)
      mutable val_attribs: attribs;

      (* Top level values have an arity inferred and/or specified in *)
      (* signatures.  The arity records the number of arguments preferred *)   
      (* in each position for a curried functions. The currying is based *)
      (* on the number of lambdas, and in each position the elements are *)
      (* based on attempting to deconstruct the type of the argument as a *)
      (* tuple-type.  The field is mutable because arities for recursive *)
      (* values are only inferred after the r.h.s. is analyzed, but the *)
      (* value itself is created before the r.h.s. is analyzed. *)
      (* *)
      (* TLR also sets this for inner bindings that it wants to represent as "top level" bindings. *)
     
      (* MUTABILITY CLEANUP: mutability of this field is used by adjustAllUsesOfRecValue *)
      (* This replaces the empty arity initially assumed with an arity garnered from the *)
      (* type-checked expression.  Consider removing.  One way to look at it is that the value *)
      (* used during type checking of the recursive binding is only a proto-value used *)
      (* to build up constraints on the generalized variable. *)
      mutable val_arity: val_arity_info;


      (** The parent type. For an extension member this is the module containing the extension *)
      val_actual_parent: parent_ref;

      (* XML documentation attached to a value. *)
      val_xmldoc : xmlDoc; 
  } 

and val_arity_info = val_arity option

(* Data carried by a member *)
and val_member_info = val_meminfo option

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  val_meminfo = 
    { vspr_il_name: string;
     
      (** The parent type. For an extension member this is the type being extended *)
      vspr_apparent_parent: tycon_ref;  

      (* Gets updated with full slotsig after interface implementation relation is checked *)
      mutable vspr_implements_slotsig: tslotsig option; 

      (** Gets updated with 'true' if an abstract slot is implemented in the file being typechecked.  Internal only. *)
      mutable vspr_implemented: bool;                      

      vspr_flags: memberFlags }


and val_inline_info =
    (** Indicates the value must always be inlined *)
    | PseudoValue 
    (** Indictes the code for the function exists, e.g. to satisfy interfaces on objects, but that it is also always inlined *)
    | AlwaysInline 
    | OptionalInline
    | NeverInline

and val_recursive_info =
    (** Set while the value is within its recursive scope. The flag indicates if the value has been eagerly generalized and accepts generic-recursive calls *)
    | ValInRecScope of bool
    (** The normal value for this flag when the value is not within its recursive scope *)
    | ValNotInRecScope

and 'a local_ref = 'a
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  'a item_ref = 
    (* Indicates a reference to something bound in this CCU *)
    | Ref_private  of 'a local_ref  
    (* Indicates a reference to something bound in another CCU *)
    | Ref_nonlocal of 'a nonlocal_item_ref

and 'a nonlocal_item_ref = 
    { (* The path to an item in another CCU *)
      nlr_nlpath : nonlocal_path; 
      (* The name of an item in another CCU *)
      nlr_item: string; 
      (* A cache to hold the data after the reference has been resolved *)
      mutable nlr_cache:  'a nonnull_slot }

(* public_path: Record where a construct lives within the global namespace. *)
and public_path      = PubPath of string list * string

(* compilation_path: The information ILXGEN needs about the location of an item *)
and compilation_path = CompPath of Il.scope_ref * (string * modul_kind) list

(* nonlocal_path: Index into the namespace/module structure of a particular CCU *)
and nonlocal_path    = NLPath of ccu * string list

and local_val_ref   = val_spec       local_ref
and local_typar_ref = typar_spec     local_ref

(* note: these are type equivalent *)
and modul_ref       = modul_spec     item_ref
and tycon_ref       = tycon_spec     item_ref

and val_ref         = val_spec       item_ref

and unionconstr_ref = UCRef of tycon_ref * string
and recdfield_ref   = RFRef of tycon_ref * string

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  typ =
  (* Indicates the type is a universal type, only used for types of values, members and record fields *)
  | TType_forall of typars * typ
  (* Indicates the type is a type application *)
  | TType_app of tycon_ref * tinst
  (* Indicates the type is a tuple type *)
  | TType_tuple of typ list
  (* Indicates the type is a function type *)
  | TType_fun of  typ * typ
  (* Indicates the type is a variable type, whether declared, generalized or an inference type parameter  *)
  | TType_var of local_typar_ref 
  (* Indicates the solution of a type variable is unknown. never used directly as a type *)
  | TType_unknown
  (* A legacy fake type used in legacy code to indicate the "type" of a module when building a module expression *)
  | TType_modul_bindings

and tinst = typ list 
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  ccu_spec = 
    { (** Holds the filename for the DLL, if any *)
      ccu_filename: string option; 
      
      (** Holds the data indicating how this assembly/module is referenced from the code being compiled. *)
      ccu_scoref: Il.scope_ref;
      
      (** A unique stamp for this DLL *)
      ccu_stamp: int;
      
      (** The fully qualified assembly reference string to refer to this assembly. This is persisted in quotations *)
      ccu_qname: string option; 
      
      (** A hint as to where does the code for the CCU live (e.g what was the tcConfig.implicitIncludeDir at compilation time for this DLL?) *)
      ccu_code_dir: string; (*  *) 
      
      (** Indicates that this DLL was compiled using the F# compiler *)
      ccu_fsharp: bool; 
      
      (** A handle to the full specification of the contents of the module contained in this ccu *)
      (* NOTE: may contain transient state during typechecking *)
      mutable ccu_contents: modul_spec }
      
and ccu = ccu_spec ccu_thunk


(*---------------------------------------------------------------------------
!* Attributes
 *------------------------------------------------------------------------- *)

and attribs = attrib list 

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  attrib_kind = 
  (** Indicates an attribute refers to a type defined in an imported .NET assembly *)
  | ILAttrib of Il.method_ref 
  (** Indicates an attribute refers to a type defined in an imported F# assembly *)
  | FSAttrib of val_ref

and attrib = 
  | Attrib of (attrib_kind * expr list * (string*typ*bool*expr) list)


(*---------------------------------------------------------------------------
!* Expressions 
 * 
 * Pattern matching has been compiled down to
 * a decision tree by this point.  The right-hand-sides (actions) of
 * the decision tree are labelled by integers that are unique for that
 * particular tree.
 *------------------------------------------------------------------------- *)

and tconst = 
  | TConst_bool       of bool
  | TConst_int8       of Nums.i8
  | TConst_uint8      of Nums.u8
  | TConst_int16      of Nums.i16
  | TConst_uint16     of Nums.u16
  | TConst_int32      of int32
  | TConst_uint32     of Nums.u32
  | TConst_int64      of int64
  | TConst_uint64     of Nums.u64
  | TConst_nativeint  of int64
  | TConst_unativeint of Nums.u64
  | TConst_float32    of Nums.ieee32
  | TConst_float      of Nums.ieee64
  | TConst_char       of Nums.unichar
  | TConst_string     of Bytes.bytes (* in unicode *)
  | TConst_bigint     of Bytes.bytes (* in unicode *)
  | TConst_decimal    of Bytes.bytes (* in unicode *)
  | TConst_bignum     of Bytes.bytes (* in unicode *)
  | TConst_unit
  | TConst_default (* null/zero-bit-pattern *)
  

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  dtree = 

    (** Indicates a decision point in a decision tree. *)
    | TDSwitch  of (* input: *) expr * (* cases: *) dtree_case list * (* default: *) dtree option * Range.range

    (** Indicates the decision tree has terminated with success, calling the given target with the given parameters *)
    | TDSuccess of (* results: *) expr list * (* target: *) int  

    (** Bind the given value throught the remaining cases of the dtree. *)
    | TDBind of (* binding: *) bind * (* body: *) dtree

and dtree_case = 
    | TCase of dtree_discrim * dtree

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  dtree_discrim = 
    (** Test if the input to a decision tree matches the given constructor *)
    | TTest_unionconstr of (unionconstr_ref * tinst) 
    (** Test if the input to a decision tree is an array of the given length *)
    | TTest_array_length of int * typ  
    (** Test if the input to a decision tree is the given constant value *)
    | TTest_const of tconst
    (** Test if the input to a decision tree is null *)
    | TTest_isnull 
    (** Test if the input to a decision tree is an instance of the given type *)
    | TTest_isinst of (* source: *) typ * (* target: *) typ
    (** Run the active pattern and bind a successful result to the (one) variable in the remaining tree *)
    (*Test if the input to a decision tree is an instance of the given type. TTest_query(expr-to-run, result-types, vref-if-named, idx-of-result, apinfo)  *)
    | TTest_query of expr * typ list * val_ref option * int * apinfo


and dtree_target = 
    (* A target of a decision tree. Can be thought of as a little function, though is compiled as a local block. *)
    | TTarget of val_spec list * expr

and binds = bind list

and bind = 
    | TBind of val_spec * expr

(* apelem_ref: active pattern element (deconstruction case), e.g. 'JNil' or 'JCons'. *)
(* Integer indicates which choice in the target set is being selected by this item. *)
and apelem_ref = APElemRef of apinfo * val_ref * int 
and apinfo = APInfo of bool * string list

and val_arity  = TopValInfo  of (* numTypars: *) int * (* args: *) topArgInfo list list * (* result: *) topArgInfo 
and topArgInfo = TopArgData of (* attributes: *) attribs * (* name: *) ident option 

and typars = typar_spec list

(* The big type of expressions.  *)
and expr =
  (** A constant expression. *)
  | TExpr_const of tconst * Range.range * typ

  (* Reference a value. The flag is only relevant if the value is an object model member *)
  (* and indicates base calls and special uses of object constructors. *)
  | TExpr_val of val_ref * val_use_flag * Range.range

  (* Sequence expressions, used for "a;b", "let a = e in b;a" and "a then b" (the last an OO constructor). *)
  | TExpr_seq of expr * expr * seq_op_kind * Range.range

  (* Lambda expressions. *)
  
  (* Why multiple vspecs? A TExpr_lambda taking multiple arguments really accepts a tuple. *)
  (* But it is in a convenient form to be compile accepting multiple *)
  (* arguments, e.g. if compiled as a toplevel static method. *)

  (* REVIEW: see if we can eliminate this and just use lambdas taking single arguments. *)
  (* though perhaps propagating metadata about preferred argument names. *)

  (* REVIEW: it would probably be better if the freevar cache cached those of the body rather than the *)
  (* whole expression.  *)

  (* REVIEW: why not conjoin multiple lambdas into a single iterated lambda node? *)
  | TExpr_lambda of int * val_spec option * val_spec list * expr * Range.range * typ * freevars cache

  (* Type lambdas.  These are used for the r.h.s. of polymorphic 'let' bindings and *)
  (* for expressions that implement first-class polymorphic values. *)
  (* REVIEW: it would probably be better if the freevar cache cached those of the body rather than the *)
  (* whole expression.  *)
  | TExpr_tlambda of int * typars * expr * Range.range * typ  * freevars cache

  (* Apoplications *)
  (* Applications combine type and term applications, and are normalized so *)
  (* that sequential applications are combined, so "(f x) y" becomes "f [x;y]". *)
  (* The type attached to the function is the formal function type, used to ensure we don't build application *)
  (* nodes that over-apply when instantiating at function types. *)
  | TExpr_app of expr * typ * tinst * expr list * Range.range

  (* Bind a recursive set of values. *)

  (* REVIEW: it would probably be better if the freevar cache cached those of the body rather than the *)
  (* whole expression.  *)
  | TExpr_letrec of binds * expr * Range.range * freevars cache

  (* Bind a value. *)
  
  (* REVIEW: do we really need both TExpr_let AND TExpr_letrec AND TExpr_match AND TExpr_lambda!? *)
  (* Why not just TExpr_match the primitive? *)
  
  (* REVIEW: it would probably be better if the freevar cache cached those of the body rather than the *)
  (* whole expression.  *)
  | TExpr_let of bind * expr * Range.range * freevars cache

  (* Object expressions: A closure that implements an interface or a base type. *)
  (* The base object type might be a delegate type. *)
  | TExpr_obj of 
       (* unique *)           int * 
       (* object typ *)       typ *                                         (* <-- NOTE: specifies type parameters for base type *)
       (* base val *)         val_spec option * 
       (* ctor call *)        expr * 
       (* overrides *)        tmethod list * 
       (* extra interfaces *) (typ * tmethod list) list *                   
                              Range.range * 
                              freevars cache

  (* Pattern matching. *)

  (* Matches are a more complicated form of "let" with multiple possible destinations *)
  (* and possibly multiple ways to get to each destination.  *)
  (* The first mark is that of the expression being matched, which is used *)
  (* as the mark for all the decision making and binding that happens during the match. *)
  | TExpr_match of Range.range * dtree * dtree_target array * Range.range * typ * freevars cache

  (* If we statically know some infomation then in many cases we can use a more optimized expression *)
  (* This is primarily used by terms in the standard library, particularly those implementing overloaded *)
  (* operators. *)
  | TExpr_static_optimization of tstatic_optimization_constraint list * expr * expr * Range.range

  (* An intrinsic applied to some (strictly evaluated) arguments *)
  (* A few of intrinsics (TOp_try, TOp_while, TOp_for) expect arguments kept in a normal form involving lambdas *)
  | TExpr_op of op_spec * tinst * expr list * Range.range

  (* Indicates the expression is a quoted expression tree. *)
  (*    - bool indicates if this is a raw (untyped) quotation or not *)
  | TExpr_quote of bool * expr * range * typ  
  
  (* TExpr_hole: used inside quotations to indicate anti-quotation points *)
  | TExpr_hole of range * typ 

  (* Typechecking residue: A free (even potentially incoherent) choice of typars that arises due to *)
  (* minimization of polymorphism at let-rec bindings.  These are *)
  (* resolved to a concrete instantiation on subsequent rewrites. *)
  (* REVIEW: give a warning whenever this choice may propagate to a construct that may inspect the *)
  (* values of runtime type variables. *)
  | TExpr_tchoose of typars * expr * Range.range

  (* Typechecking residue: A TExpr_link occurs for every use of a recursively bound variable. While type-checking *)
  (* the recursive bindings a dummy expression is stored in the mutable reference cell. *)
  (* After type checking the bindings this is replaced by a use of the variable, perhaps at an *)
  (* appropriate type instantiation. These are immediately eliminated on subsequent rewrites. *)
  | TExpr_link of expr ref

(** A type for a module-or-namespace-fragment and the actual definition of the module-or-namespace-fragment*)
and modul_expr = 
    | TMTyped of 
         (* The module_typ is a binder. However it is not used in the modul_def: it is only referenced from the 'outside' *) 
         modul_typ 
         * modul_def
         * Range.range

(** The contents of a module-or-namespace-fragment definition *)
and modul_def = 
    (** Indicates the module is a module with a signature *)
    | TMAbstract of modul_expr
    (** Indicates the module fragment is made of several module fragments in succession *)
    | TMDefs     of modul_def list  
    (** Indicates the module fragment is a 'let' definition *)
    | TMDefLet   of bind * Range.range
    (** Indicates the module fragment is a 'rec' definition of types and values *)
    | TMDefRec   of tycon_spec list * bind list * Range.range
    (** Indicates the module fragment defines a module *)
    | TMDefModul of modul_bind

(** A named module-or-namespace-fragment definition *)
and modul_bind = 
    | TMBind of 
         (** This is a 'fake' tycon_spec that represents the compilation of a module as a class. *)
         (** It carries the ident, attributes, mkind etc. of a module. It does not carry the 'contents' *)
         (** of the module, which are implicit from the modul_def, and/or factored off separately into a modul_typ. *)
         tycon_spec * 
         (** This is the body of the module/namespace *)
         modul_def

and typedImplFile = TImplFile of qualifiedNameOfFile * modul_expr

and typedAssembly = TAssembly of typedImplFile list

and recordConstructionInfo = 
   (* We're in a constructor. The purpose of the record expression is to *)
   (* fill in the fields of a pre-created but uninitialized object *)
   | RecdExprIsObjInit
   (* Normal record construction *)
   | RecdExpr
   
and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  op_spec =
  | TOp_uconstr of unionconstr_ref 
  | TOp_exnconstr of tycon_ref
  | TOp_tuple 
  | TOp_array
  (* Constant bytes, but a new mutable blob is generated each time the construct is executed *)
  | TOp_bytes of Bytes.bytes 
  (* REVIEW: simplify these two to a more general concretization of inner letrec bindnigs *)
  | TOp_while 
  | TOp_for of bool (* count up or down? *)
  | TOp_try_catch | TOp_try_finally 

  (* Construct a record or object-model value. The val_ref is for self-referential class constructors, otherwise *)
  (* it indicates that we're in a constructor and the purpose of the expression is to *)
  (* fill in the fields of a pre-created but uninitialized object, and to assign the initialized *)
  (* version of the object into the optional mutable cell pointed to be the given value. *)
  | TOp_recd of recordConstructionInfo * tycon_ref
  
  | TOp_field_set of recdfield_ref 
  | TOp_field_get of recdfield_ref 
  | TOp_field_get_addr of recdfield_ref       
  | TOp_constr_tag_get of tycon_ref 
  | TOp_constr_field_get of unionconstr_ref * int 
  | TOp_constr_field_set of  unionconstr_ref * int
  | TOp_exnconstr_field_get of tycon_ref * int 
  | TOp_exnconstr_field_set of tycon_ref * int 
  | TOp_tuple_field_get of int 
  (* IL assembly code - type list are the types pushed on the stack *)
  | TOp_asm of Il.instr list * typ list 
  | TOp_get_ref_lval (* generate a ldflda on an 'a ref. REVIEW: generalize to a TOp_flda *)
  | TOp_coerce (* Conversion node, compiled via type-directed translation or to box/unbox *)
  (* Pseudo method calls.  These are where the actual target *)
  (* of the call cannot be resolved until the F# code has been inlined *)
  (* and the type parameters resolved, i.e. these are compiled by *)
  (* doing a type-directed translation in the backend. This is used for *)
  (* overloaded operations like op_Addition. *)
  | TOp_trait_call of trait_constraint_info  (* The specification of the member constraint that will have been solved *)
  | TOp_lval_op of lval_op_kind * val_ref (* C-style operations on byrefs and mutable vals (l-values) *)
  (* IL method calls *)
  | TOp_ilcall of 
      (bool * (* virtual call? *)
       bool * (* protected? *)
       bool * (* is the object a value type? *) 
       bool * (* newobj call? *) 
       val_use_flag * (* isSuperInit call? *) 
       bool * (* property? used for reflection *)
       bool * (* DllImport? if so don't tailcall *)
       (typ * typ) option * (* coercion to box 'this' *)  
       Il.method_ref) * 
     typ list * (* tinst *) 
     typ list * (* minst *) 
     typ list   (* types of pushed values if any *) 

and lval_op_kind = 
  | LGetAddr      (* In C syntax this is: &localv            *)
  | LByrefGet     (* In C syntax this is: *localv_ptr        *)
  | LSet          (* In C syntax this is:  localv = e     , note == *(&localv) = e == LGetAddr; LByrefSet*)
  | LByrefSet     (* In C syntax this is: *localv_ptr = e   *)

and seq_op_kind = 
  | NormalSeq     (* a ; b *)
  | ThenDoSeq     (* let res = a in b;res *)  

and val_use_flag =
  | NormalValUse
  | CtorValUsedAsSuperInit
  | CtorValUsedAsSelfInit
  | VSlotDirectCall
  
and tstatic_optimization_constraint = 
  | TTyconEqualsTycon of typ * typ
  
(* A representation of a method in an object expression. Methods associated with types *)
(* use "vspr" values (see above). *)
(* Note: We should probably use vspr infos for object expressions, as then the treatment of members *)
(* in object expressions could be unified with the treatment of members in types *)
and tmethod = 
  TMethod of (tslotsig * typars * val_spec list * expr * range)
and tslotsig = TSlotSig of (string * typ * typars * typars * tslotparam list * typ)
and tslotparam = TSlotParam of  (string option * typ * bool (* in *) * bool (* out *) * bool (* optional *) * attribs)

and structVals = 
 { modVals: (string,val_ref) Map.t;
   modMods: (string,structVals) Map.t }


(*---------------------------------------------------------------------------
!* Freevars.  Computed and cached by later phases (never computed type checking).  Cached in terms. Not pickled.
 *-------------------------------------------------------------------------*)

and free_locvals = local_val_ref Zset.t
and free_loctypars = local_typar_ref Zset.t
and free_loctycons = tycon_spec Zset.t
and free_rfields = recdfield_ref Zset.t
and free_uconstrs = unionconstr_ref Zset.t
and free_tyvars = 
    { (* The summary of locally defined type definitions used in the expression. These may be made private by a signature *)      
      (* and we have to check various conditions associated with that. *)
      free_loctycons: free_loctycons;
      
      (* The summary of type parameters used in the expression. These may not escape the enclosing generic construct *)
      (* and we have to check various conditions associated with that. *)
      free_loctypars: free_loctypars }

and freevars = 
    { (* The summary of locally defined variables used in the expression. These may be hidden at let bindings etc. *)
      (* or made private by a signature or marked 'internal' or 'private', and we have to check various conditions associated with that. *)
      free_locvals: free_locvals;
      
      (* Indicates if the expression contains a call to a protected member or a base call. *)
      (* Calls to protected members and direct calls to super classes can't escape, also code can't be inlined *)
      uses_method_local_constructs: bool; 

      (* The summary of locally defined tycon representations used in the expression. These may be made private by a signature *)      
      (* or marked 'internal' or 'private' and we have to check various conditions associated with that. *)
      free_loctycon_reprs: free_loctycons; 

      (* The summary of fields used in the expression. These may be made private by a signature *)      
      (* or marked 'internal' or 'private' and we have to check various conditions associated with that. *)
      free_rfields: free_rfields;
      
      (* The summary of union constructors used in the expression. These may be *)      
      (* marked 'internal' or 'private' and we have to check various conditions associated with that. *)
      free_uconstrs: free_uconstrs;
      
      (* See free_tyvars above. *)
      free_tyvars: free_tyvars }

(*---------------------------------------------------------------------------
!* Compiled representations of type and exception definitions.  
 * Computed and cached by later phases (never computed type checking).  Cached at 
 * type and exception definitions. Not pickled.
 *-------------------------------------------------------------------------*)

and il_type_repr = 
  | TyrepNamed of Il.type_ref * Il.boxity
  | TyrepOpen of Il.typ  

(*---------------------------------------------------------------------------
!* Basic ops on locally defined things (guaranteed not to be references to
 * other assemblies/compilation-units)
 *-------------------------------------------------------------------------*)

let deref_local (lv : 'a osgn local_ref) = (lv : 'a osgn)
let derefd_local lv = deref_osgn (deref_local lv)
let deref_local_typar  (lv : local_typar_ref) = deref_local lv
let deref_local_val    (lv : local_val_ref)   = deref_local lv
let derefd_local_typar (lv : local_typar_ref) = derefd_local lv

(*---------------------------------------------------------------------------
!* Equality relations on locally defined things 
 *-------------------------------------------------------------------------*)

let local_vref_eq   lv1 lv2 = (derefd_local lv1).val_stamp   =!= (derefd_local lv2).val_stamp
let local_tcref_eq  lv1 lv2 = (derefd_local lv1).tycon_stamp =!= (derefd_local lv2).tycon_stamp
let typar_ref_eq    lv1 lv2 = (derefd_local lv1).typar_stamp =!= (derefd_local lv2).typar_stamp

(*---------------------------------------------------------------------------
!* Basic properties on pe definitions
 *-------------------------------------------------------------------------*)

let data_of_tycon (v:tycon_spec) = deref_osgn v 
(* we're cross compiling, otherwise we'd use F# properties for these! *)
let nested_of_tycon x           = Lazy.force ((data_of_tycon x).tycon_modul_contents)
let tcaug_of_tycon   tc = (data_of_tycon tc).tycon_tcaug
let name_of_tycon           tc = (data_of_tycon tc).tycon_name
let range_of_tycon          tc = (data_of_tycon tc).tycon_range
let id_of_tycon             tc = ident(name_of_tycon tc, range_of_tycon tc)
let repr_of_tycon           tc = (data_of_tycon tc).tycon_repr
let typars_of_tycon         tc = (data_of_tycon tc).tycon_typars
let abbrev_of_tycon         tc = (data_of_tycon tc).tycon_abbrev
let stamp_of_tycon          tc = (data_of_tycon tc).tycon_stamp
let repr_access_of_tycon    tc = (data_of_tycon tc).tycon_repr_access
let attribs_of_tycon        tc = (data_of_tycon tc).tycon_attribs
let xmldoc_of_tycon         tc = (data_of_tycon tc).tycon_xmldoc
let il_repr_cache_of_tycon  tc = (data_of_tycon tc).tycon_il_repr_cache
let pubpath_of_tycon        tc = (data_of_tycon tc).tycon_pubpath
let access_of_tycon        tc = (data_of_tycon tc).tycon_access
let prefix_display_of_tycon tc = (data_of_tycon tc).tycon_prefix_display
let tycon_is_modul          tc = (data_of_tycon tc).tycon_is_modul
let cpathopt_of_tycon x     = (data_of_tycon x).tycon_cpath 
let cpath_of_tycon x        = match cpathopt_of_tycon x with Some cpath -> cpath | None -> error(Error("type/module "^name_of_tycon x^" is not a concrete module",range_of_tycon x))

let submoduls_of_mtyp x = x.mtyp_submoduls
let tycons_of_mtyp x = x.mtyp_tycons
let mkind_of_mtyp    x = x.mtyp_kind
let all_submoduls_of_mtyp x = Namemap.range (submoduls_of_mtyp x)

let data_of_modul (v:modul_spec) = deref_osgn v 
let mtyp_of_modul x           = Lazy.force ((data_of_modul x).tycon_modul_contents)
let mkind_of_modul x          = x |> mtyp_of_modul |> mkind_of_mtyp
let id_of_modul x             = ident((data_of_modul x).tycon_name,(data_of_modul x).tycon_range)
let xmldoc_of_modul x         = (data_of_modul x).tycon_xmldoc
let attribs_of_modul x        = (data_of_modul x).tycon_attribs
let stamp_of_modul x          = (data_of_modul x).tycon_stamp
let pubpath_of_modul x        = (data_of_modul x).tycon_pubpath
let access_of_modul x         = (data_of_modul x).tycon_access
let cpath_of_modul x          = cpath_of_tycon x
let range_of_modul x          = (id_of_modul x).idRange
let name_of_modul x           = (id_of_modul x).idText


let mtyp_tryfind_tycon n x = Map.tryfind n (tycons_of_mtyp x)
let mtyp_has_submodul n x = Map.mem n (submoduls_of_mtyp x)
let mtyp_get_submodul n x = 
    try Map.find n (submoduls_of_mtyp x) 
    with Not_found -> errorR(Failure("Internal error: module '"^n^"' not found")); raise Not_found
let mtyp_tryfind_submodul n x = Map.tryfind n (submoduls_of_mtyp x)

(*---------------------------------------------------------------------------
!* Flags on values
 *-------------------------------------------------------------------------*)

module ValSpecFlags = struct

    let base_of_vflags x =
                                  match (x.val_flags &&&&      0b0000000000000110L) with 
                                                             | 0b0000000000000000L -> BaseVal
                                                             | 0b0000000000000010L -> CtorThisVal
                                                             | 0b0000000000000100L -> NormalVal
                                                             | 0b0000000000000110L -> MemberThisVal
                                                             | _          -> failwith "base_of_vflags"

    let encode_base_of_vflags x val_flags =
                     (val_flags &&&&                       ~~~~0b0000000000000110L)
                                |||| (match x with
                                        | BaseVal ->           0b0000000000000000L
                                        | CtorThisVal ->       0b0000000000000010L
                                        | NormalVal ->         0b0000000000000100L
                                        | MemberThisVal ->     0b0000000000000110L)



    let compgen_of_vflags x =           (x.val_flags &&&&      0b0000000000001000L) <> 0x0L
    let encode_compgen_of_vflags b val_flags = 
                         if b then      (  val_flags ||||      0b0000000000001000L) 
                         else           (  val_flags &&&& ~~~~ 0b0000000000001000L)
    let mustinline_of_vflags x =
                                  match (x.val_flags &&&&      0b0000000000110000L) with 
                                                             | 0b0000000000000000L -> PseudoValue
                                                             | 0b0000000000010000L -> AlwaysInline
                                                             | 0b0000000000100000L -> OptionalInline
                                                             | 0b0000000000110000L -> NeverInline
                                                             | _          -> failwith "mustinline_of_vflags"

    let encode_mustinline_of_vflags x val_flags =
                     (val_flags &&&&                       ~~~~0b0000000000110000L)
                                |||| (match x with
                                        | PseudoValue ->       0b0000000000000000L
                                        | AlwaysInline ->      0b0000000000010000L
                                        | OptionalInline ->    0b0000000000100000L
                                        | NeverInline ->       0b0000000000110000L)

    let mutability_of_vflags x =
                                  match (x.val_flags &&&&      0b0000000001000000L) with 
                                                             | 0b0000000000000000L -> Immutable
                                                             | 0b0000000001000000L -> Mutable
                                                             | _          -> failwith "mutability_of_vflags"

    let encode_mutability_of_vflags x val_flags =
                     (val_flags &&&&                       ~~~~0b0000000001000000L)
                                |||| (match x with
                                        | Immutable ->         0b0000000000000000L
                                        | Mutable   ->         0b0000000001000000L)

    let modbind_of_vflags x =
                                  match (x.val_flags &&&&      0b0000000010000000L) with 
                                                             | 0b0000000000000000L -> false
                                                             | 0b0000000010000000L -> true
                                                             | _          -> failwith "modbind_of_vflags"

    let encode_modbind_of_vflags x val_flags =
                     (val_flags &&&&                       ~~~~0b0000000010000000L)
                                |||| (match x with
                                        | false     ->         0b0000000000000000L
                                        | true      ->         0b0000000010000000L)

    let isext_of_vflags x =
                                  match (x.val_flags &&&&      0b0000000100000000L) with 
                                                             | 0b0000000000000000L -> false
                                                             | 0b0000000100000000L -> true
                                                             | _          -> failwith "isext_of_vflags"

    let encode_isext_of_vflags x val_flags =
                     (val_flags &&&&                       ~~~~0b0000000100000000L)
                                |||| (match x with
                                        | false     ->         0b0000000000000000L
                                        | true      ->         0b0000000100000000L)
    let is_implicit_ctor_of_vflags x =
                                  match (x.val_flags &&&&      0b0000001000000000L) with 
                                                             | 0b0000000000000000L -> false
                                                             | 0b0000001000000000L -> true
                                                             | _          -> failwith "is_implicit_ctor_of_vflags"

    let encode_is_implicit_ctor_of_vflags x val_flags =
                     (val_flags &&&&                       ~~~~0b0000001000000000L)
                                |||| (match x with
                                        | false     ->         0b0000000000000000L
                                        | true      ->         0b0000001000000000L)

    let is_tyfunc_of_vflags x =
                                  match (x.val_flags &&&&      0b0000010000000000L) with 
                                                             | 0b0000000000000000L -> false
                                                             | 0b0000010000000000L -> true
                                                             | _          -> failwith "is_implicit_ctor_of_vflags"

    let encode_is_tyfunc_of_vflags x val_flags =
                     (val_flags &&&&                       ~~~~0b0000010000000000L)
                                |||| (match x with
                                        | false     ->         0b0000000000000000L
                                        | true      ->         0b0000010000000000L)

    let vrec_of_vflags x =       match  (x.val_flags &&&&      0b0001100000000000L) with 
                                                             | 0b0000000000000000L -> ValNotInRecScope
                                                             | 0b0000100000000000L -> ValInRecScope(true)
                                                             | 0b0001000000000000L -> ValInRecScope(false)
                                                             | _                   -> failwith "vrec_of_vflags"
    let encode_vrec_of_vflags x val_flags =
                    (val_flags &&&&                        ~~~~0b0001100000000000L)
                               |||| (match x with
                                     | ValNotInRecScope     -> 0b0000000000000000L
                                     | ValInRecScope(true)  -> 0b0000100000000000L
                                     | ValInRecScope(false) -> 0b0001000000000000L)

    let encode_val_flags (vrec,base,compgen,mustinline,mut,isModuleBinding,isExtensionMember,isImplicitCtor,isTyFunc) =
        0L |> encode_vrec_of_vflags      vrec 
           |> encode_base_of_vflags       base
           |> encode_compgen_of_vflags    compgen
           |> encode_mustinline_of_vflags mustinline
           |> encode_mutability_of_vflags mut
           |> encode_modbind_of_vflags    isModuleBinding
           |> encode_isext_of_vflags      isExtensionMember
           |> encode_is_implicit_ctor_of_vflags      isImplicitCtor
           |> encode_is_tyfunc_of_vflags      isTyFunc
end

(*---------------------------------------------------------------------------
!* Metadata on values (names of arguments etc. 
 *-------------------------------------------------------------------------*)


module TopValData = struct
  let namedTupleTopArg id = TopArgData([],Some id)
  let unnamedTopArg1 = TopArgData([],None)
  let unnamedTopArg = [unnamedTopArg1]
  let unitArgData = unnamedTopArg
  let unnamedRetVal = TopArgData([],None)
  let selfMetadata = unnamedTopArg
  let emptyTopValData = TopValInfo(0,[],unnamedRetVal)

  let hasNoArgs (TopValInfo(n,args,_)) = (n = 0) && isNil args
  let numCurriedArgs(TopValInfo(_,args,_)) = List.length args
  let getArgInfos(TopValInfo(_,args,_)) = args
  let aritiesOfArgs (TopValInfo(_,args,_)) = List.map List.length args
end

(*---------------------------------------------------------------------------
!* Basic properties on values 
 *-------------------------------------------------------------------------*)

let data_of_val (v:val_spec) = deref_osgn v 
let type_of_val       v = (data_of_val v).val_type
let access_of_val     v = (data_of_val v).val_access
let defn_range_of_val v = (data_of_val v).val_defn_range
let name_of_val       v = (data_of_val v).val_name
let literal_const_of_val       v = (data_of_val v).val_const
let range_of_val      v = (data_of_val v).val_range
let id_of_val         v = ident(name_of_val v,range_of_val v)
let val_is_toplevel   v = (data_of_val v).val_arity |> isSome
let pubpath_of_val    v = (data_of_val v).val_pubpath
let const_of_val      v = (data_of_val v).val_const
let published_closed_defn_of_val v = (data_of_val v).val_defn
let member_info_of_val       v = (data_of_val v).val_meminfo
let mutability_of_val v = (data_of_val v) |> ValSpecFlags.mutability_of_vflags
let modbind_of_val    v = (data_of_val v) |> ValSpecFlags.modbind_of_vflags
let isext_of_val      v = (data_of_val v) |> ValSpecFlags.isext_of_vflags
let is_implicit_ctor_of_val      v = (data_of_val v) |> ValSpecFlags.is_implicit_ctor_of_vflags
let vrec_of_val       v = (data_of_val v) |> ValSpecFlags.vrec_of_vflags
let base_of_val       v = (data_of_val v) |> ValSpecFlags.base_of_vflags
let is_tyfunc_of_val       v = (data_of_val v) |> ValSpecFlags.is_tyfunc_of_vflags
let arity_of_val      v = (data_of_val v).val_arity
let stamp_of_val      v = (data_of_val v).val_stamp
let inlineFlag_of_val v = (data_of_val v) |> ValSpecFlags.mustinline_of_vflags
let compgen_of_val    v = (data_of_val v) |> ValSpecFlags.compgen_of_vflags
let attribs_of_val    v = (data_of_val v).val_attribs
let xmldoc_of_val     v = (data_of_val v).val_xmldoc
  
let arity2_of_val v = (match arity_of_val v with None -> TopValData.emptyTopValData | Some arities -> arities)

let actual_parent_of_val v = (data_of_val v).val_actual_parent 

let actual_parent_of_vspr_val v = 
    match actual_parent_of_val v  with 
    | Parent tcref -> tcref
    | _ -> error(InternalError("actual_parent_of_vspr_val: does not have a parent",range_of_val v))
        
let apparent_parent_of_vspr_val v = 
    match member_info_of_val v with 
    | Some vspr -> vspr.vspr_apparent_parent
    | None -> error(InternalError("apparent_parent_of_vspr_val",range_of_val v))
        
let set_vrec_of_vflags       x b = x.val_flags <- ValSpecFlags.encode_vrec_of_vflags       b x.val_flags
let set_base_of_vflags       x b = x.val_flags <- ValSpecFlags.encode_base_of_vflags       b x.val_flags
let set_mustinline_of_vflags x b = x.val_flags <- ValSpecFlags.encode_mustinline_of_vflags b x.val_flags


(*---------------------------------------------------------------------------
!* Basic properties on exception definitions
 *-------------------------------------------------------------------------*)

let is_exception_name nm = has_suffix nm "Exception"
let demangle_exception_name nm = if is_exception_name nm then drop_suffix nm "Exception" else nm
    

(* we're cross compiling, otherwise we'd use F# properties for these! *)
let data_of_exnc (v:tycon_spec) = deref_osgn v 
let exn_repr_of_tycon          x = (data_of_exnc x).tycon_exnc_info
let tycon_is_exnc  x = match exn_repr_of_tycon x with TExnNone -> false | _ -> true
let demangled_name_of_exnc          x = 
    let nm = (name_of_tycon x) in 
    if tycon_is_exnc x then demangle_exception_name nm else nm 
    
let attribs_of_exnc       x = (data_of_exnc x).tycon_attribs
let typars_of_exnc        x = []  (* REVIEW: allow exception constructors *) 
let xmldoc_of_exnc        x = (data_of_exnc x).tycon_xmldoc
let pubpath_of_exnc       x = (data_of_exnc x).tycon_pubpath
let access_of_exnc        x = (data_of_exnc x).tycon_access
let cpath_of_exnc         x = cpath_of_tycon x
let augmentation_of_exnc  x = (data_of_exnc x).tycon_tcaug
let il_repr_cache_of_exnc x = (data_of_exnc x).tycon_il_repr_cache

(*---------------------------------------------------------------------------
!* Basic properties on union constructors (tags)
 *-------------------------------------------------------------------------*)

let attribs_of_uconstr uc = uc.uconstr_attribs
let range_of_uconstr   uc = uc.uconstr_id.idRange
let id_of_uconstr      uc = uc.uconstr_id
let access_of_uconstr  uc = uc.uconstr_access
let name_of_uconstr    uc = (id_of_uconstr uc).idText

(*---------------------------------------------------------------------------
!* Basic properties on record/class fields
 *-------------------------------------------------------------------------*)

let access_of_rfield   v = v.rfield_access
let pattribs_of_rfield v = v.rfield_pattribs
let fattribs_of_rfield v = v.rfield_fattribs
let range_of_rfield    v = v.rfield_id.idRange
let id_of_rfield       v = v.rfield_id
let name_of_rfield     v = v.rfield_id.idText
let secret_of_rfield   v = v.rfield_secret
let static_of_rfield   v = v.rfield_static

(*---------------------------------------------------------------------------
!* Basic properties on type variables 
 *-------------------------------------------------------------------------*)

(* encode typar flags into a bit field  *)
type typar_rigidity = TyparRigid | TyparWarnIfNotRigid | TyparFlexible
let rigid_of_tpdata x =            
                              match (x.typar_flags &&&     0b1100000l) with 
                                                         | 0b0000000l -> TyparRigid
                                                         | 0b0100000l -> TyparWarnIfNotRigid
                                                         | 0b1000000l -> TyparFlexible
                                                         | _          -> failwith "rigid_of_tpdata"
let encode_rigid_of_tpdata x typar_flags =
                 (typar_flags &&&                       ~~~0b1100000l)
                              ||| (match x with
                                    | TyparRigid          -> 0b0000000l
                                    | TyparWarnIfNotRigid -> 0b0100000l
                                    | TyparFlexible       -> 0b1000000l)

let from_error_of_tpdata x =        (x.typar_flags &&&     0b0000010l) <> 0x0l
let encode_from_error_of_tpdata b typar_flags = 
                     if b then      (  typar_flags |||     0b0000010l) 
                     else           (  typar_flags &&& ~~~ 0b0000010l)

let compgen_of_tpdata x =           (x.typar_flags &&&     0b0000100l) <> 0x0l
let encode_compgen_of_tpdata b typar_flags = 
                     if b then      (  typar_flags |||     0b0000100l) 
                     else           (  typar_flags &&& ~~~ 0b0000100l)
let static_req_of_tpdata x =
                              match (x.typar_flags &&&     0b0011000l) with 
                                                         | 0b0000000l -> NoStaticReq
                                                         | 0b0001000l -> HeadTypeStaticReq
                                                         | 0b0010000l -> CompleteStaticReq
                                                         | _          -> failwith "static_req_of_tpdata"

let encode_static_req_of_tpdata x typar_flags =
                 (typar_flags &&&                       ~~~0b0011000l)
                              ||| (match x with
                                    | NoStaticReq ->       0b0000000l
                                    | HeadTypeStaticReq -> 0b0001000l
                                    | CompleteStaticReq -> 0b0010000l)

let set_rigid_of_tpdata      x b = x.typar_flags <- encode_rigid_of_tpdata      b x.typar_flags
let set_from_error_of_tpdata x b = x.typar_flags <- encode_from_error_of_tpdata b x.typar_flags
let set_static_req_of_tpdata x b = x.typar_flags <- encode_static_req_of_tpdata b x.typar_flags

let encode_typar_flags (rigid,error,compgen,static_req) =
    0l |> encode_rigid_of_tpdata      rigid 
       |> encode_from_error_of_tpdata error
       |> encode_compgen_of_tpdata    compgen
       |> encode_static_req_of_tpdata static_req


let data_of_typar (v:typar_spec) = deref_osgn v 
let name_of_typar        tc = (data_of_typar tc).typar_id.idText
let range_of_typar       tc = (data_of_typar tc).typar_id.idRange
let id_of_typar          tc = (data_of_typar tc).typar_id
let stamp_of_typar       tc =  (data_of_typar tc).typar_stamp
let solution_of_typar    tc =  (data_of_typar tc).typar_solution
let constraints_of_typar tc =  (data_of_typar tc).typar_constraints
let compgen_of_typar     tc =  (data_of_typar tc) |> compgen_of_tpdata
let rigid_of_typar       tc =  (data_of_typar tc) |> rigid_of_tpdata
let static_req_of_typar  tc =  (data_of_typar tc) |> static_req_of_tpdata
let from_error_of_typar  tc =  (data_of_typar tc) |> from_error_of_tpdata

let typ_of_slotparam (TSlotParam(_,ty,_,_,_,_)) = ty


(*---------------------------------------------------------------------------
!* Basic properties on an entire compilation unit
 *-------------------------------------------------------------------------*)

let name_of_ccu           ccu = name_of_ccu_thunk ccu
let scoref_of_ccu         ccu = (deref_ccu_thunk ccu).ccu_scoref
let stamp_of_ccu          ccu = (deref_ccu_thunk ccu).ccu_stamp
let filename_of_ccu       ccu = (deref_ccu_thunk ccu).ccu_filename
let qualified_name_of_ccu ccu = (deref_ccu_thunk ccu).ccu_qname
let code_dir_of_ccu       ccu = (deref_ccu_thunk ccu).ccu_code_dir
let ccu_is_fsharp         ccu = (deref_ccu_thunk ccu).ccu_fsharp
let top_modul_of_ccu      ccu = (deref_ccu_thunk ccu).ccu_contents
let top_moduls_of_ccu     ccu = submoduls_of_mtyp (mtyp_of_modul (top_modul_of_ccu ccu))
let top_tycons_of_ccu     ccu = tycons_of_mtyp (mtyp_of_modul (top_modul_of_ccu ccu))

(*---------------------------------------------------------------------------
!* Aggregate operations to help transform the components that 
 * make up the entire compilation unit
 *-------------------------------------------------------------------------*)

let mapTImplFile      f   (TImplFile(fragName,moduleExpr)) = TImplFile(fragName, f moduleExpr)
let fmapTImplFile     f z (TImplFile(fragName,moduleExpr)) = let z,moduleExpr = f z moduleExpr in z,TImplFile(fragName,moduleExpr)
let map_acc_TImplFile f z (TImplFile(fragName,moduleExpr)) = let moduleExpr,z = f z moduleExpr in TImplFile(fragName,moduleExpr), z
let foldTImplFile     f z (TImplFile(fragName,moduleExpr)) = f z moduleExpr

(*---------------------------------------------------------------------------
!* Dereferencing
 *-------------------------------------------------------------------------*)

let adjust_module_name istype nm = (match istype with AsMangledNamedType str -> nm^str | _ -> nm)

exception UndefinedName of int * string * ident * string list
exception InternalUndefinedTyconItem of string * tycon_ref * string
exception InternalUndefinedItemRef of string * nonlocal_path * string

let try_deref_modul_in_modul modul nm = 
    let mtyp = mtyp_of_modul modul in
    match mtyp_tryfind_submodul nm mtyp with 
    | Some _ as res -> res
    | None -> 
          (* We hack of the "Module" when resolving paths coming from import.ml type_ref's - see tycon_of_il_tref in import.ml *)
          match try_drop_suffix nm "Module" with 
          | Some modName when mtyp_has_submodul modName mtyp -> mtyp_tryfind_submodul modName mtyp
          (* Structures nested in a tycon effectively reference the tycon as if it were a module *)
          | _ -> mtyp_tryfind_tycon nm mtyp 

let rec deref_path_in_modul nlpath modul p = 
    match p with 
    | [] -> modul
    | h :: t -> 
        let next = try_deref_modul_in_modul modul h in
        match next with 
        | Some res -> deref_path_in_modul nlpath res t
        | None -> 
            (errorR (InternalUndefinedItemRef ("module/namespace",nlpath, h)); raise Not_found) 

let rec try_deref_path_in_modul nlpath modul p = 
    match p with 
    | [] -> Some modul
    | h :: t -> 
        let next = try_deref_modul_in_modul modul h in
        match next with 
        | Some res -> try_deref_path_in_modul nlpath res t
        | None -> None
        

let try_modul_of_nlpath (NLPath(ccu,p) as nlpath) = try_deref_path_in_modul  nlpath (top_modul_of_ccu ccu) p
let modul_of_nlpath (NLPath(ccu,p) as nlpath) = deref_path_in_modul  nlpath (top_modul_of_ccu ccu) p
let try_mtyp_of_nlpath x = x |> try_modul_of_nlpath |> Option.map mtyp_of_modul
let mtyp_of_nlpath x = x |> modul_of_nlpath |> mtyp_of_modul

(* REVIEW: This should be removed in favour of equality over val_refs *)
let vspec_eq (lv1: val_spec) (lv2: val_spec) = (lv1 =!= lv2)
let ccu_eq (mv1: ccu) (mv2: ccu) = (mv1 =!= mv2) || (top_modul_of_ccu mv1 =!= top_modul_of_ccu mv2)
let tpspec_eq (tp1: typar_spec) (tp2: typar_spec) = (tp1 =!= tp2)

let (*F# inline F#*) nlr_cached nlr f = 
  (*IF-OCAML*) match     nlr.nlr_cache with None -> let res = f() in nlr.nlr_cache <- nullable_slot_full res; res | Some x -> x             (*ENDIF-OCAML*)
  (*F#         match box nlr.nlr_cache with null -> let res = f() in nlr.nlr_cache <- nullable_slot_full res; res |      _ -> nlr.nlr_cache F#*)


let encode_modref_name nm = "/"^nm
let is_encoded_modref_name nm = String.length nm > 0 && nm.[0] = '/'
let decode_modref_name nm = String.sub nm 1 (String.length nm - 1)
let try_decode_modref_name nm = if is_encoded_modref_name nm then decode_modref_name nm else nm

let deref_tycon (tcr :tycon_ref) = 
    match tcr with 
    | Ref_private x -> deref_local x
    | Ref_nonlocal nlr ->
        nlr_cached nlr (fun () -> 
          let modul = mtyp_of_nlpath nlr.nlr_nlpath in 
          if is_encoded_modref_name nlr.nlr_item then 
              let moduleName = decode_modref_name nlr.nlr_item in
              try Map.find moduleName modul.mtyp_submoduls
              with Not_found -> 
                  raise (InternalUndefinedItemRef ("module or namespace",nlr.nlr_nlpath, moduleName))
          else
              match Map.tryfind nlr.nlr_item modul.mtyp_tycons with 
              | Some res -> res
              | None -> 
                  raise (InternalUndefinedItemRef ("type",nlr.nlr_nlpath, nlr.nlr_item)))
        
let deref_modul (tcr :modul_ref) = deref_tycon tcr

let deref_nlval nlr =
    nlr_cached nlr (fun () -> 
        let modul = mtyp_of_nlpath nlr.nlr_nlpath in  
        try Map.find nlr.nlr_item modul.mtyp_vals
        with Not_found -> raise (InternalUndefinedItemRef ("val",nlr.nlr_nlpath, nlr.nlr_item)))
      
let vals_of_mtyp x = x.mtyp_vals
let try_deref_nlval nlr = try_mtyp_of_nlpath nlr.nlr_nlpath |> Option.bind (vals_of_mtyp >> Map.tryfind nlr.nlr_item)

let deref_val (vr :val_ref) = 
    match vr with 
    | Ref_private x -> deref_local_val x
    | Ref_nonlocal nlr -> (deref_nlval nlr)  
      
let try_deref_val (vr :val_ref) = 
    match vr with 
    | Ref_private x -> Some (deref_local_val x)
    | Ref_nonlocal nlr -> try_deref_nlval nlr
        
let is_il_tycon x = match repr_of_tycon x with | Some (TIlObjModelRepr _) -> true |  _ -> false
let is_il_tcref   x = x |> deref_tycon |> is_il_tycon
let dest_il_tycon x = match repr_of_tycon x with | Some (TIlObjModelRepr (a,b,c)) -> (a,b,c) |  _ -> failwith "dest_il_tcref"
let dest_il_tcref x = x |> deref_tycon |> dest_il_tycon

let is_union_tycon tycon = match repr_of_tycon tycon with | Some (TFiniteUnionRepr _) -> true |  _ -> false
let is_union_tcref tcr = is_union_tycon (deref_tycon tcr)

let is_recd_tycon tcr = match repr_of_tycon tcr with | Some (TRecdRepr _) -> true |  _ -> false
let is_recd_tcref tcr = is_recd_tycon (deref_tycon tcr)

let is_fsobjmodel_tycon tcr = match repr_of_tycon tcr with | Some (TFsObjModelRepr _) -> true |  _ -> false
let is_fsobjmodel_tcref tcr = is_fsobjmodel_tycon (deref_tycon tcr)

let is_abstract_tycon tycon = match abbrev_of_tycon tycon,repr_of_tycon tycon with | None,None -> true |  _ -> false
let is_abstract_tcref tcr = is_abstract_tycon (deref_tycon tcr)

let rfield_by_idx rfields n = 
   if n >= 0 && n < Array.length rfields.rfields_by_index then rfields.rfields_by_index.(n) 
   else failwith "rfield_by_idx"

let rfield_by_name x n = Namemap.tryfind n x.rfields_by_name

let uconstr_by_idx uconstrs n = 
   if n >= 0 && n < Array.length uconstrs.uconstrs_by_index then uconstrs.uconstrs_by_index.(n) 
   else failwith "uconstr_by_idx"

let all_rfields_of_rfield_tables x = x.rfields_by_index |> Array.to_list
let true_rfields_of_rfield_tables x = x |> all_rfields_of_rfield_tables |> List.filter (secret_of_rfield >> not)   
let uconstrs_of_uconstr_tables x = x.uconstrs_by_index |> Array.to_list
let rfields_array_of_uconstr uc = uc.uconstr_rfields.rfields_by_index
let rfields_of_uconstr uc = uc.uconstr_rfields |> all_rfields_of_rfield_tables
let rfield_of_uconstr_by_idx uc n = rfield_by_idx uc.uconstr_rfields n
let rfield_of_uconstr_by_name uc nm = rfield_by_name uc.uconstr_rfields nm
let is_nullary_of_uconstr uc = (Array.length (rfields_array_of_uconstr uc)  = 0)

let mk_uconstrs_table ucs = 
    { uconstrs_by_index = Array.of_list ucs; 
      uconstrs_by_name = Namemap.of_keyed_list name_of_uconstr ucs }
                                                                  

let mk_rfields_table ucs = 
    { rfields_by_index = Array.of_list ucs; 
      rfields_by_name = Namemap.of_keyed_list name_of_rfield ucs }
                                                                  
let rfields_of_tycon tycon = 
    match (data_of_tycon tycon).tycon_repr with 
    | Some (TRecdRepr x | TFsObjModelRepr {fsobjmodel_rfields=x}) -> x
    |  _ -> 
    match exn_repr_of_tycon tycon with 
    | TExnFresh x -> x
    | _ -> mk_rfields_table []

let rfields_array_of_tycon tycon = (rfields_of_tycon tycon).rfields_by_index
let rfield_of_tycon_by_idx tycon n = rfield_by_idx (rfields_of_tycon tycon) n
let any_rfield_of_tycon_by_name tycon n = rfield_by_name (rfields_of_tycon tycon) n
let tycon_objmodel_data_of_tycon tycon = 
   match (data_of_tycon tycon).tycon_repr with 
   | Some (TFsObjModelRepr x) -> x 
   |  _ -> failwith "super_of_tycon_data"

let funion_of_tycon d = match (data_of_tycon d).tycon_repr with | Some (TFiniteUnionRepr x) -> Some x |  _ -> None

let all_rfields_of_tycon        x = x |> rfields_array_of_tycon |> Array.to_list
let instance_rfields_of_tycon   x = x |> all_rfields_of_tycon |> List.filter (fun f -> not f.rfield_static && not f.rfield_secret)
let all_rfields_of_tcref        x = x |> deref_tycon |> all_rfields_of_tycon
let instance_rfields_of_tcref   x = x |> deref_tycon |> instance_rfields_of_tycon
let rfields_array_of_tcref      x = x |> deref_tycon |> rfields_array_of_tycon
let rfield_of_tcref_by_idx      x = x |> deref_tycon |> rfield_of_tycon_by_idx
let any_rfield_of_tcref_by_name x = x |> deref_tycon |> any_rfield_of_tycon_by_name

let uconstrs_array_of_funion x = x.funion_constrs.uconstrs_by_index 
let uconstrs_of_funion       x = x |> uconstrs_array_of_funion |> Array.to_list

let uconstrs_array_of_tycon tycon = 
    match funion_of_tycon tycon with 
    | Some x -> uconstrs_array_of_funion x
    | None -> [| |] 
let uconstrs_array_of_tcref x = x |> deref_tycon |> uconstrs_array_of_tycon

let uconstrs_of_tycon x = x |> uconstrs_array_of_tycon |> Array.to_list
let uconstrs_of_tcref x = x |> uconstrs_array_of_tcref |> Array.to_list

let uconstr_of_tycon_by_idx tycon n = 
    match funion_of_tycon tycon with 
    | Some x -> uconstr_by_idx x.funion_constrs n
    | _ -> failwith "uconstr_of_tycon_by_idx"
let uconstr_of_tcref_by_idx x = x |> deref_tycon |> uconstr_of_tycon_by_idx

let uconstr_of_tycon_by_name tycon n = 
    match funion_of_tycon tycon with 
    | Some x  -> Namemap.tryfind n x.funion_constrs.uconstrs_by_name
    | None -> None
let uconstr_of_tcref_by_name x = x |> deref_tycon |> uconstr_of_tycon_by_name

let tycon_objmodel_data_of_tcref x = x |> deref_tycon |> tycon_objmodel_data_of_tycon

let is_abbrev_tcref tc = isSome (abbrev_of_tycon (deref_tycon tc))
          
let deref_rfield (RFRef(tcref,id)) = 
    let tycon = deref_tycon tcref in 
    tycon,
    match any_rfield_of_tcref_by_name tcref id with 
    | Some res -> res
    | None -> raise (InternalUndefinedTyconItem ("field",tcref, id))

let deref_uconstr (UCRef(tcref,id)) = 
    let tycon = deref_tycon tcref in 
    tycon,
    match uconstr_of_tcref_by_name tcref id with 
    | Some res -> res
    | None -> raise (InternalUndefinedTyconItem ("constr",tcref, id))

let rec deref_exnc (ecr :tycon_ref) = deref_tycon ecr


(*--------------------------------------------------------------------------
!* Make references to TAST items
 *------------------------------------------------------------------------ *)

let mk_rfref tcref f = RFRef(tcref, f)
let mk_ucref tcref c = UCRef(tcref, c)
let rfref_of_rfield tcref f = mk_rfref tcref f.rfield_id.idText
let ucref_of_uconstr tcref c = mk_ucref tcref c.uconstr_id.idText
let mk_nlpath (NLPath(mref,p)) n = NLPath(mref,p@[n])
let mk_cpath (CompPath(scoref,p)) n istype = CompPath(scoref,p@[(n,istype)])


(*---------------------------------------------------------------------------
!* Get information from refs
 *--------------------------------------------------------------------------*)

let solution_of_tpref    x = x |> deref_local_typar |> solution_of_typar 
let stamp_of_tpref       x = x |> deref_local_typar |> stamp_of_typar 
let name_of_tpref        x = x |> deref_local_typar |> name_of_typar 
let constraints_of_tpref x = x |> deref_local_typar |> constraints_of_typar 
let compgen_of_tpref     x = x |> deref_local_typar |> compgen_of_typar 
let rigid_of_tpref       x = x |> deref_local_typar |> rigid_of_typar 
let static_req_of_tpref  x = x |> deref_local_typar |> static_req_of_typar 
let range_of_tpref       x = x |> deref_local_typar |> range_of_typar 

let type_of_vref             v = v |> deref_val |> type_of_val 
let id_of_vref               v = v |> deref_val |> id_of_val 
let name_of_vref             v = v |> deref_val |> name_of_val 
let range_of_vref            v = v |> deref_val |> range_of_val 
let mutability_of_vref       v = v |> deref_val |> mutability_of_val 
let pubpath_of_vref          v = v |> deref_val |> pubpath_of_val 
let base_of_vref             v = v |> deref_val |> base_of_val 
let is_tyfunc_of_vref        v = v |> deref_val |> is_tyfunc_of_val 
let vrec_of_vref             v = v |> deref_val |> vrec_of_val 
let arity_of_vref            v = v |> deref_val |> arity_of_val 
let stamp_of_vref            v = v |> deref_val |> stamp_of_val 
let inlineFlag_of_vref       v = v |> deref_val |> inlineFlag_of_val 
let isext_of_vref            v = v |> deref_val |> isext_of_val 
let is_implicit_ctor_of_vref v = v |> deref_val |> is_implicit_ctor_of_val 
let compgen_of_vref          v = v |> deref_val |> compgen_of_val 
let attribs_of_vref          v = v |> deref_val |> attribs_of_val 
let member_info_of_vref             v = v |> deref_val |> member_info_of_val 
let xmldoc_of_vref           v = v |> deref_val |> xmldoc_of_val
let apparent_parent_of_vspr_vref  v = v |> deref_val |> apparent_parent_of_vspr_val
let actual_parent_of_vspr_vref    v = v |> deref_val |> actual_parent_of_vspr_val
let actual_parent_of_vref    v = v |> deref_val |> actual_parent_of_val
 
let stamp_of_lvref      v = v |> deref_local_val |> stamp_of_val 
let type_of_lvref       v = v |> deref_local_val |> type_of_val 
let base_of_lvref       v = v |> deref_local_val |> base_of_val 
let mutability_of_lvref v = v |> deref_local_val |> mutability_of_val 
let vspr_of_lvref       v = v |> deref_local_val |> member_info_of_val 

let name_of_tcref           x = x |> deref_tycon |> name_of_tycon 
let range_of_tcref          x = x |> deref_tycon |> range_of_tycon 
let stamp_of_tcref          x = x |> deref_tycon |> stamp_of_tycon 
let repr_access_of_tcref    x = x |> deref_tycon |> repr_access_of_tycon 
let attribs_of_tcref        x = x |> deref_tycon |> attribs_of_tycon 
let typars_of_tcref         x = x |> deref_tycon |> typars_of_tycon 
let tcaug_of_tcref          x = x |> deref_tycon |> tcaug_of_tycon
let prefix_display_of_tcref x = x |> deref_tycon |> prefix_display_of_tycon
let il_repr_cache_of_tcref  x = x |> deref_tycon |> il_repr_cache_of_tycon
let pubpath_of_tcref        x = x |> deref_tycon |> pubpath_of_tycon 
let xmldoc_of_tcref         x = x |> deref_tycon |> xmldoc_of_tycon 

let demangled_name_of_ecref    x = x |> deref_exnc |> demangled_name_of_exnc

let tcref_of_rfref (RFRef(tcref,id)) = tcref
let name_of_rfref (RFRef(tcref,id)) = id
let tycon_of_rfref    x = x |> tcref_of_rfref |> deref_tycon
let rfield_of_rfref   x = x |> deref_rfield |> snd
let pattribs_of_rfref x = x |> rfield_of_rfref |> pattribs_of_rfield 
let range_of_rfref    x = x |> rfield_of_rfref |> range_of_rfield
        
let tcref_of_ucref (UCRef(tcref,id)) = tcref
let name_of_ucref (UCRef(tcref,id)) = id
let tycon_of_ucref   x = x |> tcref_of_ucref |> deref_tycon
let uconstr_of_ucref x = x |> deref_uconstr |> snd
let attribs_of_ucref x = x |> uconstr_of_ucref |> attribs_of_uconstr 
let range_of_ucref   x = x |> uconstr_of_ucref |> range_of_uconstr 

let path_of_nlpath (NLPath(a,b)) = b
let ccu_of_nlpath  (NLPath(a,b)) = a

let mtyp_of_modref    x = x |> deref_modul |> mtyp_of_modul
let pubpath_of_modref x = x |> deref_modul |> pubpath_of_modul
let mkind_of_modref  x = x |> mtyp_of_modref |> mkind_of_mtyp
let xmldoc_of_modref  x = x |> deref_modul |> xmldoc_of_modul
let name_of_modref    x = x |> deref_modul |> name_of_modul 
let range_of_modref   x = x |> deref_modul |> range_of_modul 
let stamp_of_modref   x = x |> deref_modul |> stamp_of_modul 

let nlpath_of_nlref nlr = nlr.nlr_nlpath 
let item_of_nlref   nlr = nlr.nlr_item
let ccu_of_nlref    nlr = ccu_of_nlpath (nlpath_of_nlref nlr)

let nlref_of_item_ref iref = 
    match iref with 
    | Ref_private _ -> None
    | Ref_nonlocal nlr -> Some nlr

let ccu_of_item_ref iref =  iref |> nlref_of_item_ref |> Option.map ccu_of_nlref

let total_of_apref (APElemRef(total,vref,n)) = total
let vref_of_apref (APElemRef(_,vref,n)) = vref
let idx_of_apref (APElemRef(_,vref,n)) = n

(*--------------------------------------------------------------------------
!* Type parameters and inference unknowns
 *-------------------------------------------------------------------------*)

let mk_typar_ty tp = TType_var tp
let copy_typar (data: typar_spec) = let x = deref_osgn data in new_osgn { x with typar_stamp=new_stamp() }
let copy_typars tps = List.map copy_typar tps

(*--------------------------------------------------------------------------
 * Inference variables
 *-------------------------------------------------------------------------- *)

let tpref_is_solved r = 
    match (solution_of_tpref r) with 
    | TType_unknown -> false
    | _ -> true
    
let try_shortcut_solved_tpref canShortcut r = 
  let ty = solution_of_tpref r in 
  if canShortcut then begin
      match ty with 
      | TType_var r2 -> 
         begin match solution_of_tpref r2 with
         | TType_unknown  -> ()
         | res -> 
            let tp = derefd_local_typar r in 
            tp.typar_solution <- res;
         end
      | _ -> () 
  end;
  ty

let rec strip_tpeqnsA canShortcut ty = 
  match ty with 
  | TType_var r when tpref_is_solved r -> strip_tpeqnsA canShortcut (try_shortcut_solved_tpref canShortcut r) 
  | _ -> ty

let strip_tpeqns ty = strip_tpeqnsA false ty

(*--------------------------------------------------------------------------
!* Construct local references
 *-------------------------------------------------------------------------- *)

let mk_lref (x: 'a) = ( (* LRef *)  x : 'a local_ref)
let mk_local_ref (x: 'a) = Ref_private (mk_lref x)
let mk_nlr mp id = {nlr_nlpath = mp; nlr_item=id; nlr_cache = nullable_slot_empty()}
let mk_nonlocal_ref mp id = Ref_nonlocal (mk_nlr mp id)

(*--------------------------------------------------------------------------
!* From Ref_private to Ref_nonlocal when exporting data.
 *------------------------------------------------------------------------ *)

let enclosing_nlpath_of_pubpath viewedCcu (PubPath(p,nm)) = NLPath(viewedCcu, p)
let nlpath_of_pubpath viewedCcu (PubPath(p,nm)) = NLPath(viewedCcu,p@[nm])
let nlpath_of_modul viewedCcu v = pubpath_of_modul v |> Option.map (nlpath_of_pubpath viewedCcu)

let nlref_of_pubpath viewedCcu (PubPath(p,nm) as pubpath) x = 
    mk_nlr (enclosing_nlpath_of_pubpath viewedCcu pubpath) nm

let rescope_val_pubpath viewedCcu pubpath x : val_ref = Ref_nonlocal (nlref_of_pubpath viewedCcu pubpath x)
let rescope_tycon_pubpath viewedCcu pubpath x : tycon_ref = Ref_nonlocal (nlref_of_pubpath viewedCcu pubpath x)
let rescope_module_pubpath viewedCcu (PubPath(p,nm) as pubpath) x : modul_ref = 
    Ref_nonlocal  (mk_nlr (enclosing_nlpath_of_pubpath viewedCcu pubpath) (encode_modref_name nm))

let mk_subref cref nm x = 
    match cref with 
    | Ref_private _ -> mk_local_ref x
    | Ref_nonlocal nlr ->
        let (NLPath(ccu,p)) = nlr.nlr_nlpath in 
        let moduleName = try_decode_modref_name nlr.nlr_item in 
        mk_nonlocal_ref (NLPath(ccu, p@[moduleName])) nm

let mk_local_tpref (x : typar_spec) = mk_lref x
let mk_local_vref (x : val_spec) = mk_lref x
  
(*---------------------------------------------------------------------------
!* Equality between TAST items.
 *--------------------------------------------------------------------------*)

let item_ref_in_this_assembly compilingFslib x = 
    match x with 
    | Ref_private _ -> true
    | Ref_nonlocal _ -> compilingFslib

let nlpath_eq (NLPath(x1,y1) as smr1 :nonlocal_path) (NLPath(x2,y2) as smr2) = 
    smr1 =!= smr2 || (ccu_eq x1 x2 && y1 = y2)

let nlref_eq nlr1 nlr2 = 
    (nlr1 =!= nlr2 ) || 
    (nlpath_eq nlr1.nlr_nlpath nlr2.nlr_nlpath && 
     (nlr1.nlr_item =!= nlr2.nlr_item  || nlr1.nlr_item  = nlr2.nlr_item))

(* Compiler-internal references to items in fslib are generated as Ref_nonlocal even when compiling fslib *)
let fslib_nlpath_eq_pubpath nlr (PubPath(path,nm)) = 
    nlr.nlr_item = nm  &&
    let (NLPath(ccu,p)) = nlr.nlr_nlpath in 
    if (p = path) then true else (warning(Failure(sprintf "%s <> %s" (String.concat ";" p) (String.concat ";" path))); false)

let fslib_refs_eq ppF namef fslibCcu x y  =
    match x,y with 
    | (Ref_nonlocal nlr, Ref_private x)
    | (Ref_private x, Ref_nonlocal nlr) ->
      ccu_eq (ccu_of_nlpath nlr.nlr_nlpath) fslibCcu &&
      let pubpathOpt = ppF x in 
      isSome pubpathOpt && fslib_nlpath_eq_pubpath nlr (the pubpathOpt)
    | (Ref_private x, Ref_private y) ->
      let pubpathOpt1 = ppF x in 
      let pubpathOpt2 = ppF y in 
      isSome pubpathOpt1 && isSome pubpathOpt2 && pubpathOpt1 = pubpathOpt2
    | _ -> false
  
let prim_tcref_eq compilingFslib fslibCcu (x : tycon_ref) (y : tycon_ref) = 
    x =!= y ||
    match x,y with 
    | Ref_private a, Ref_private b when local_tcref_eq a b -> true
    | Ref_nonlocal nlr1, Ref_nonlocal nlr2 when nlref_eq nlr1 nlr2 -> true
    | _ -> compilingFslib && fslib_refs_eq pubpath_of_tycon name_of_tycon fslibCcu x y  

let prim_ucref_eq compilingFslib fslibCcu (UCRef(tcr1,c1) as uc1) (UCRef(tcr2,c2) as uc2) = 
    uc1 =!= uc2 || (prim_tcref_eq compilingFslib fslibCcu tcr1 tcr2 && c1 = c2)

let prim_vref_eq compilingFslib fslibCcu (x : val_ref) (y : val_ref) =
    x =!= y ||
    match x,y with 
    | Ref_private a, Ref_private b when local_vref_eq a b -> true
    | Ref_nonlocal nlr1, Ref_nonlocal nlr2 when nlref_eq nlr1 nlr2 -> true
    | _ -> compilingFslib && fslib_refs_eq pubpath_of_val  name_of_val fslibCcu x y
 
(*---------------------------------------------------------------------------
 * pubpath/cpath mess
 *------------------------------------------------------------------------- *)

let name_of_scoref sref = 
    match sref with 
    | Il.ScopeRef_local -> "<local>"
    | Il.ScopeRef_module mref -> mref.Il.modulRefName
    | Il.ScopeRef_assembly aref -> aref.Il.assemRefName
  
let access_path_of_cpath (CompPath(scoref,cpath))  = cpath
let path_of_cpath (CompPath(scoref,path))  = List.map fst path
let text_of_cpath (CompPath(scoref,path)) = name_of_scoref scoref ^"/"^ (text_of_path (List.map fst path))
let access_count_of_cpath (CompPath(scoref,cpath))  = length (List.filter (function (_,Namespace) -> false | _ -> true) cpath)
let pubpath_of_cpath id cpath = PubPath(path_of_cpath cpath,id.idText)
let parent_cpath (CompPath(scoref,cpath)) = 
    let a,b = frontAndBack cpath in 
    CompPath(scoref,a)
let full_cpath_of_modul m = 
    let (CompPath(scoref,cpath))  = cpath_of_modul m in 
    CompPath(scoref,cpath@[(name_of_modul m, mkind_of_mtyp (mtyp_of_modul m))])

let can_access_cpath_from (CompPath(scoref1,cpath1)) (CompPath(scoref2,cpath2)) = 
    List.length cpath1 <= List.length cpath2 &&
    (cpath1 = front (List.length cpath1) cpath2) &&
    (scoref1 = scoref2) 
let can_access_from (TAccess x) cpath = 
    x |> List.for_all (fun cpath1 -> can_access_cpath_from cpath1 cpath)
let can_access_from_everywhere (TAccess x) = isNil x
let can_access_from_somewhere (TAccess x) = true
let isLessAccessible (TAccess aa) (TAccess bb)  = 
   (* not (gen_subset_of (=) aa bb) *)
    not (aa |> List.for_all(fun a -> bb |> List.exists (fun b -> can_access_cpath_from a b)))
let string_of_access (TAccess paths) = String.concat ";" (List.map text_of_cpath paths)


let cpath_of_ccu ccu = CompPath(scoref_of_ccu ccu,[]) 
let nlpath_of_ccu ccu = NLPath(ccu,[]) 

(*---------------------------------------------------------------------------
 * Construct TAST nodes
 *------------------------------------------------------------------------- *)

let new_ccu nm x  : ccu = new_ccu_thunk nm x

let new_typar (rigid,Typar(id,var,compgen),error,attribs) = 
    new_osgn
      { typar_id = id; 
        typar_stamp = new_stamp(); 
        typar_flags= encode_typar_flags (rigid,error,compgen,var); 
        typar_attribs= attribs; 
        typar_solution = TType_unknown;
        typar_constraints=[];
        typar_xmldoc = emptyXMLDoc; (* todo *) } 

let mk_rigid_typar nm m = new_typar (TyparRigid,Typar(mksyn_id m nm,NoStaticReq,true),false,[])
let new_tcaug () =  { tcaug_compare=None; tcaug_equals=None; tcaug_structural_hash=None; 
                      tcaug_hasObjectGetHashCode=false; 
                      tcaug_adhoc=Namemap.empty_multi; tcaug_super=None;tcaug_implements=[]; 
                      tcaug_closed=false; tcaug_abstract=false; }

let taccessPublic = TAccess []
let combineAccess (TAccess a1) (TAccess a2) = TAccess(a1@a2)

let new_uconstr id nm tys rty attribs docOption vis = 
    { uconstr_id=id;
      uconstr_il_name=nm;
      uconstr_xmldoc=docOption;
      uconstr_access=vis;
      uconstr_rfields = mk_rfields_table tys;
      uconstr_rty = rty;
      uconstr_attribs=attribs } 

let set_tcaug_compare tcaug x = tcaug.tcaug_compare         <- Some x
let set_tcaug_equals  tcaug x = tcaug.tcaug_equals          <- Some x
let set_tcaug_hash    tcaug x = tcaug.tcaug_structural_hash <- Some x
let set_tcaug_hasObjectGetHashCode tcaug b = tcaug.tcaug_hasObjectGetHashCode <- b

let compute_actual_val_parent specialRepr isExtensionMember parent =  
     if not(isExtensionMember) && isSome(specialRepr) then parent else ParentNone

let new_mtype istypename tycons vals = 
    let moduls,tycons = List.partition tycon_is_modul tycons in 
    { mtyp_kind=istypename;
      mtyp_submoduls=Namemap.of_keyed_list name_of_tycon moduls;
      mtyp_tycons= Namemap.of_keyed_list name_of_tycon tycons;
      mtyp_vals=Namemap.of_keyed_list name_of_val vals;
      mtyp_apref_cache = ref None;
      mtyp_tyconsByDemangledNameAndArity_cache = ref None;
      mtyp_tyconsByAccessNames_cache = ref None ;
      mtyp_exconsByDemangledName_cache = ref None }

let empty_mtype istypename = new_mtype istypename [] []

let new_exnc cpath id vis repr attribs doc = 
    let id = mksyn_id id.idRange (id.idText^"Exception") in
    new_osgn
      { tycon_stamp=new_stamp();
        tycon_attribs=attribs;
        tycon_name=id.idText;
        tycon_range=id.idRange;
        tycon_exnc_info= repr;
        tycon_tcaug=new_tcaug();
        tycon_xmldoc=doc;
        tycon_pubpath=cpath |> Option.map (pubpath_of_cpath id);
        tycon_access=vis;
        tycon_repr_access=vis;
        tycon_modul_contents = notlazy (empty_mtype AsNamedType);
        tycon_cpath= cpath;
        tycon_typars=[];
        tycon_abbrev = None;
        tycon_repr = None;
        tycon_prefix_display=false; (* TODO, though note these are not generic anyway *)
        tycon_is_modul = false;
        tycon_il_repr_cache= new_cache() ; } 

let new_rfield  stat konst id ty mut pattribs fattribs docOption vis secret =
    { rfield_mutable=mut;
      rfield_pattribs=pattribs;
      rfield_fattribs=fattribs;
      rfield_type=ty;
      rfield_static=stat;
      rfield_const=konst;
      rfield_access = vis;
      rfield_secret = secret;
      rfield_xmldoc = docOption; 
      rfield_id=id; }

    
let new_tycon cpath (nm,m) vis repr_vis typars docOption preferPostfix mtyp =
    let stamp = new_stamp() in 
    if verbose then dprintf2 "new_tycon, id = %s, stamp #%d\n" nm stamp;
    new_osgn
      { tycon_stamp=stamp;
        tycon_name=nm;
        tycon_range=m;
        tycon_prefix_display=preferPostfix;
        tycon_attribs=[];
        tycon_typars=typars;
        tycon_abbrev = None;
        tycon_repr = None;
        tycon_repr_access = repr_vis;
        tycon_exnc_info=TExnNone;
        tycon_tcaug=new_tcaug();
        tycon_modul_contents = mtyp;
        tycon_access=vis;
        tycon_xmldoc = docOption;
        tycon_pubpath=cpath |> Option.map (pubpath_of_cpath (mksyn_id m nm));
        tycon_cpath = cpath;
        tycon_is_modul =false;
        tycon_il_repr_cache = new_cache(); }

let mk_il_type_def_tycon nlpath id tps (scoref,enc,tdef) mtyp =
    let tycon = new_tycon nlpath id taccessPublic taccessPublic tps emptyXMLDoc true mtyp in 
    (deref_osgn tycon).tycon_repr <- Some (TIlObjModelRepr (scoref,enc,tdef));
    (tcaug_of_tycon tycon).tcaug_closed <- true;
    tycon

exception Duplicate of string * ident
exception FullAbstraction of string * range

let mk_namemap s idf items = 
    fold_right (fun item sofar -> 
        if Namemap.mem (idf item).idText sofar then raise (Duplicate(s,idf item));
        Namemap.add (idf item).idText item sofar) items Namemap.empty

let mk_tycon_namemap     = mk_namemap "type"      id_of_tycon
let mk_exnconstr_namemap = mk_namemap "exception" id_of_tycon
let mk_val_namemap       = mk_namemap "value"     id_of_val

let new_mspec cpath vis id xml attribs mtype  = 
  let stamp = new_stamp() in 
  if verbose then dprintf2 "new_mspec, id = %s, stamp #%d\n" id.idText stamp;
  new_osgn 
    { tycon_name=id.idText;
      tycon_range = id.idRange;
      tycon_stamp=stamp;
      tycon_modul_contents = mtype;
      tycon_prefix_display=false; 
      tycon_is_modul =true;
      tycon_typars=[];
      tycon_abbrev = None;
      tycon_repr = None;
      tycon_repr_access = vis;
      tycon_exnc_info=TExnNone;
      tycon_tcaug=new_tcaug();
      tycon_pubpath=cpath |> Option.map (pubpath_of_cpath id);
      tycon_cpath=cpath;
      tycon_access=vis;
      (* modul_placeholder=placeholder; *) 
      tycon_attribs=attribs;
      tycon_xmldoc=xml;
      tycon_il_repr_cache = new_cache(); }

let new_vspec (id,ty,mut,compgen,arity,cpathOpt,vis,vrec,specialRepr,base,attribs,mustinline,doc,isModuleBinding,isExtensionMember,isImplicitCtor,isTyFunc,konst,parent) : val_spec = 
    let stamp = new_stamp() in 
    if verbose then dprintf2 "new_vspec, id = %s, stamp #%d\n" id.idText stamp;
    new_osgn { val_stamp = stamp;
               val_name=id.idText;
               val_range=id.idRange;
               val_defn_range=id.idRange;
               val_defn=None;
               val_arity = arity;
               val_actual_parent=compute_actual_val_parent specialRepr isExtensionMember parent;
               val_flags = ValSpecFlags.encode_val_flags(vrec,base,compgen,mustinline,mut,isModuleBinding,isExtensionMember,isImplicitCtor,isTyFunc);
               val_pubpath= cpathOpt |> Option.map (pubpath_of_cpath id);
               val_const= konst;
               val_access=vis;
               val_meminfo=specialRepr;
               val_attribs=attribs;
               val_type = ty;
               val_xmldoc = doc; } 

let mk_funion ucs = {funion_constrs=mk_uconstrs_table ucs; funion_ilx_repr=new_cache()}
let mk_TFiniteUnionRepr ucs = TFiniteUnionRepr (mk_funion ucs)

(*--------------------------------------------------------------------------
!* Cloning and adjusting
 *------------------------------------------------------------------------ *)
 
(* Create a tycon based on an existing one using the function 'f'. *)
(* We require that we be given the new parent for the new tycon. *)
(* We pass the new tycon to 'f' in case it needs to reparent the *)
(* contents of the tycon. *)
let new_tycon_modified f orig = 
    let stamp = new_stamp() in 
    let data = (data_of_tycon orig) in
    if verbose then dprintf2 "new_tycon_modified, stamp #%d, based on stamp #%d\n" stamp data.tycon_stamp;
    new_osgn (f { data with tycon_stamp=stamp; }) 
    
(* Create a module tycon_spec based on an existing one using the function 'f'. *)
(* We require that we be given the parent for the new module. *)
(* We pass the new module to 'f' in case it needs to reparent the *)
(* contents of the module. *)
let new_mspec_modified f orig = 
    orig |> new_tycon_modified (fun d -> 
        { d with tycon_modul_contents = lazy (f (Lazy.force d.tycon_modul_contents)) }) 


(* Create a val_spec based on an existing one using the function 'f'. *)
(* We require that we be given the parent for the new val_spec. *)
let new_vspec_modified f orig = 
    let data = data_of_val orig in 
    let stamp = new_stamp() in 
    if verbose then dprintf2 "new_vspec_modified, stamp #%d, based on stamp #%d\n" stamp data.val_stamp;
    let data' = f { data with val_stamp=stamp } in
    new_osgn data'

let copy_mspec orig =  new_mspec_modified (fun mty -> mty) orig
let copy_tycon orig =  new_tycon_modified (fun d -> d) orig

(*--------------------------------------------------------------------------
!* Combine module types when multiple namespace fragments contribute to the
 * same namespace, making new module specs as we go.
 *------------------------------------------------------------------------ *)

let combine_maps f m1 m2 = 
  Map.fold (fun k v acc -> Map.add k (if Map.mem k m2 then f [v;Map.find k m2] else f [v]) acc) m1 
    (Map.fold (fun k v acc -> if Map.mem k m1 then acc else Map.add k (f [v]) acc) m2 Map.empty)

let combine_xmldoc (XMLDoc l1) (XMLDoc l2) = XMLDoc (l1@l2)

let rec combine_msigtyps m mty1 mty2  = 
    match mty1.mtyp_kind,mty2.mtyp_kind  with 
    | Namespace,Namespace -> 
        { mtyp_kind= (if mty1.mtyp_kind=mty2.mtyp_kind then mty1.mtyp_kind else error(Error("namespace/module mismatch",m)));
          mtyp_submoduls=combine_maps combine_mspecl mty1.mtyp_submoduls mty2.mtyp_submoduls ;
          mtyp_vals  =combine_maps (function [] -> failwith "??" | [v] -> v | v :: _ -> errorR(Error( "duplicate values named "^name_of_val v^" in two namespace fragments",range_of_val v)); v) mty1.mtyp_vals mty2.mtyp_vals ;
          mtyp_tycons=combine_maps (function [] -> failwith "??" | [v] -> v | v :: _ -> errorR(Error( "duplicate type definitions in two namespace fragments",range_of_tycon v)); v) mty1.mtyp_tycons mty2.mtyp_tycons ;
          mtyp_apref_cache = ref None;
          mtyp_tyconsByDemangledNameAndArity_cache = ref None;
          mtyp_tyconsByAccessNames_cache = ref None;
          mtyp_exconsByDemangledName_cache = ref None }
    | Namespace, _ | _,Namespace -> error(Error("namespace/module mismatch between two parts of this assembly",m))
    | AsMangledNamedType _, AsNamedType
    | AsNamedType, AsMangledNamedType _  -> error(Error("module attribute mismatch between two parts of this assembly: one module has a mangled name and another doesn't",m))
    | AsMangledNamedType _, AsMangledNamedType _
    | AsNamedType, AsNamedType -> error(InternalError("two modules in the different parts of the program have the same name",m))

and combine_mspecl l = 
    match l with
    | h :: t -> List.fold_left combine_mspecs (copy_mspec h) t
    | _ -> failwith "combine_mspecl"

and combine_mtyps m l = 
    match l with
    | h :: t -> List.fold_left (combine_msigtyps m) h t
    | _ -> failwith "combine_mspecl"

and combine_mspecs mspec1 mspec2 = 

    let m = range_of_modul mspec2 in 
    let data2 = data_of_modul mspec2 in 

    mspec1 |> new_tycon_modified (fun data1 -> 
                { data1 with 
                     tycon_xmldoc = combine_xmldoc data1.tycon_xmldoc data2.tycon_xmldoc;
                     tycon_attribs = data1.tycon_attribs @ data2.tycon_attribs;
                     tycon_modul_contents=lazy (combine_msigtyps m (mtyp_of_modul mspec1) (mtyp_of_modul mspec2)); }) 
