/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.internal.model;

import static org.junit.Assert.assertTrue;

import org.junit.Test;
import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Import;
import org.springframework.config.java.internal.parsing.CircularImportException;
import org.springframework.config.java.internal.parsing.ConfigurationParser;

import test.common.beans.TestBean;


/**
 * TCK-style unit tests for handling circular use of the {@link Import} annotation. Explore subclass
 * hierarchy for specific concrete implementations.
 *
 * @author  Chris Beams
 */
public abstract class AbstractCircularImportDetectionTests {
    protected abstract ConfigurationParser newParser(ConfigurationModel model);
    protected abstract Object loadAsConfigurationSource(Class<?> clazz) throws Exception;

    @Test
    public void simpleCircularImportIsDetected() throws Exception {
        boolean threw = false;
        try {
            newParser(new ConfigurationModel()).parse(loadAsConfigurationSource(A.class));
        } catch (CircularImportException ex) {
            assertTrue("Wrong message. Got: " + ex.getMessage(),
                       ex.getMessage().contains("Illegal attempt by @Configuration class 'AbstractCircularImportDetectionTests.B' to import class 'AbstractCircularImportDetectionTests.A'"));
            threw = true;
        }

        assertTrue(threw);
    }


    @Test
    public void complexCircularImportIsDetected() throws Exception {
        boolean threw = false;
        try {
            newParser(new ConfigurationModel()).parse(loadAsConfigurationSource(X.class));
        } catch (CircularImportException ex) {
            assertTrue("Wrong message. Got: " + ex.getMessage(),
                       ex.getMessage().contains("Illegal attempt by @Configuration class 'AbstractCircularImportDetectionTests.Z2' to import class 'AbstractCircularImportDetectionTests.Z'"));
            threw = true;
        }

        assertTrue(threw);
    }

    @Import(B.class)
    static class A {
        @Bean
        TestBean b1() { return new TestBean(); }
    }

    @Import(A.class)
    static class B {
        @Bean
        TestBean b2() { return new TestBean(); }
    }

    @Import({ Y.class, Z.class })
    class X {
        @Bean
        TestBean x() { return new TestBean(); }
    }

    class Y {
        @Bean
        TestBean y() { return new TestBean(); }
    }

    @Import({ Z1.class, Z2.class })
    class Z {
        @Bean
        TestBean z() { return new TestBean(); }
    }

    class Z1 {
        @Bean
        TestBean z1() { return new TestBean(); }
    }

    @Import(Z.class)
    class Z2 {
        @Bean
        TestBean z2() { return new TestBean(); }
    }
}
