/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.internal.model;

import static org.junit.Assert.assertEquals;

import org.junit.Test;

import org.springframework.config.java.internal.util.AnnotationExtractionUtils;
import org.springframework.config.java.internal.util.MethodAnnotationPrototype;

import java.lang.annotation.Annotation;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;


/**
 * Unit tests for {@link JavaConfigMethod}.
 *
 * @author  Chris Beams
 */
public class JavaConfigMethodTests {

    private Foo foo = AnnotationExtractionUtils.extractMethodAnnotation(Foo.class, new MethodAnnotationPrototype() {
            @Foo
            public void targetMethod() { }
        }.getClass());

    private Bar bar = AnnotationExtractionUtils.extractMethodAnnotation(Bar.class, new MethodAnnotationPrototype() {
            @Bar
            public void targetMethod() { }
        }.getClass());

    private JavaConfigMethod method =
        new JavaConfigMethod("test", 0, new Annotation[] { foo }) {
            @Override
            public Annotation getMetadata() { return foo; }
        };


    // -------------------------------------------------------------------------
    // UsageError tests
    // -------------------------------------------------------------------------

    @Test
    public void testIncompatibleAnnotationError() {
        String actual = method.new IncompatibleAnnotationError(bar).getDescription();
        String expected =
            "@Foo method 'test' is not compatible with @Bar annotation. "
            + "Remove the @Bar annotation to continue.";

        assertEquals(expected, actual);
    }

    @Test
    public void testPrivateMethodError() {
        String actual = method.new PrivateMethodError().getDescription();
        String expected =
            "@Foo method 'test' may not be private. If bean hiding is desired, "
            + "consider making this method protected or package-private";

        assertEquals(expected, actual);
    }

    @Test
    public void testFinalMethodError() {
        String actual = method.new FinalMethodError().getDescription();
        String expected = "@Foo method 'test' may not be final. Remove the final modifier to continue.";

        assertEquals(expected, actual);
    }

    @Retention(RetentionPolicy.RUNTIME)
    private @interface Foo { }

    @Retention(RetentionPolicy.RUNTIME)
    private @interface Bar { }

}
