##
#   Alexander-Schleicher ASW-28 Electrical System
#	by Bea Wolf (D-ECHO) 10/2023

#	adapted from: Cessna 172p Electrical System


var volts = 0.0;

var bus_volts = 0.0;

var ammeter_ave = 0.0;

# Initialize properties
var electrical	=	props.globals.initNode("/systems/electrical");
var serviceable	=	electrical.initNode("serviceable", 1, "BOOL");
var batt_prop	=	electrical.initNode("battery");
var output_prop	=	electrical.initNode("outputs");
var cb_prop	=	props.globals.initNode("/controls/circuit-breakers");

var volts	=	electrical.initNode("volts", 0.0, "DOUBLE");
var amps	=	electrical.initNode("amps",  0.0, "DOUBLE");

var switches = {
	master:	props.globals.getNode("/controls/electric/battery-switch"),
};

var radio	=	output_prop.initNode("comm[0]", 0.0, "DOUBLE");
var xpdr	=	output_prop.initNode("transponder", 0.0, "DOUBLE");
var flarm	=	output_prop.initNode("flarm", 0.0, "DOUBLE");
var s80	=	output_prop.initNode("S80", 0.0, "DOUBLE");

var comm_ptt	=	props.globals.getNode("/instrumentation/comm[0]/ptt");
var xpdr_mode	=	props.globals.initNode("/instrumentation/transponder/inputs/knob-mode", 0, "INT");
var xpdr_brt	=	props.globals.getNode("/instrumentation/transponder/brightness", 1);

# Array of circuit breakers
var circuit_breakers = {
	flarm: cb_prop.initNode("flarm", 1, "BOOL"),
	vario: cb_prop.initNode("vario", 1, "BOOL"),
	comm:  cb_prop.initNode("comm", 1, "BOOL"),
	xpdr:	 cb_prop.initNode("transponder", 1, "BOOL"),
};

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

############################
####	Battery Packs	####
############################

##	glider battery:	Sonnenschein dryfit A 212/5,7S (ref. MM, p. 2.16) -> no more available
##				use replacement https://airbatt.de/EXIDE-SONNENSCHEIN-Blei-Gel-Dryfit-Solar-S12-66-S-12V-66Ah
##				ideal volts: 12.0
##				ideal amps: 0.06
##				amp hours: 6.6
##				charge amps: 25

var	battery = BatteryClass.new( 12.0, 0.06, 6.6, 25, 0);

var reset_battery = func {
	battery.reset_to_full_charge();
}
var reset_circuit_breakers = func {
	# Reset circuit breakers
	foreach(var cb; keys(circuit_breakers)){
		circuit_breakers[cb].setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_bus(dt);
	}
};


var update_bus = func (dt) {
	if( !serviceable.getBoolValue() ){
		return;
	}
	var load = 0.0;
	var bus_volts = 0.0;
	
	var master_pos = switches.master.getIntValue();
	
	if( switches.master.getBoolValue() ){
		bus_volts = battery.get_output_volts();
	}
	
	# switch state
	load += cockpit_bus( bus_volts );
	
	if ( load > 300 ) {
		circuit_breakers.master.setBoolValue( 0 );
	}
	
	battery.apply_load( load, dt );
}

#Load sources:
#	com:			https://www.becker-avionics.com/wp-content/uploads/00_Literature/6200_Series_IO_en.pdf, p.2	7.5A fuse (p. 89)
#	xpdr:			https://www.becker-avionics.com/wp-content/uploads/2018/06/BXP6401_IO_issue05.pdf, p.16	3A fuse (p.16)
#	S80 vario:		https://www.lx-avionik.de/wp/download/manuals/S8x-S10xManualEnglishVer908rev57.pdf		3A fuse (p.123)
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf				0.5A fuse (p. 9)
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM

var cockpit_bus = func( bv ) {
	var load = 0.0;
	var bus_volts = bv;
	
	# Electrical Consumers (Instruments)
	if( bus_volts > 9 ){	# minimal working voltage
		
		# Radio
		if( circuit_breakers.comm.getBoolValue() ){
			radio.setDoubleValue( bus_volts );
			if( comm_ptt.getBoolValue() ){
				load += 18 / bus_volts; # transmitting: <= 2 A at 12V
			}else{
				load += 1.56 / bus_volts; # standby: <= 140 mA at 12V
			}
		} else {
			radio.setDoubleValue( 0.0 );
		}


		# XPDR Power
		if( xpdr_mode.getIntValue() != 0 and circuit_breakers.xpdr.getBoolValue() and bus_volts != 0 ){
			xpdr.setDoubleValue( bus_volts );
			if( xpdr_mode.getIntValue() > 1 ){
				load += ( 0.37 + 0.03 * xpdr_brt.getDoubleValue() ) * 14.0 / bus_volts;	# 0.37 - 0.40 A at 14VDC depending on display brightness setting
			} else {
				load += ( 0.22 + 0.06 * xpdr_brt.getDoubleValue() ) * 14.0 / bus_volts;	# 0.22 - 0.28 A at 14VDC depending on display brightness setting
			}
		} else {
			xpdr.setDoubleValue( 0.0 );
		}

		# Vario
		if( circuit_breakers.vario.getBoolValue() ){
			s80.setDoubleValue( bus_volts );
			load += 1.980 / bus_volts; # 140mA - 190mA depending on display brightness setting at 12V
		} else {
			s80.setDoubleValue( 0.0 );
		}
		
	} else {
		radio.setDoubleValue( 0.0 );
		xpdr.setDoubleValue( 0.0 );
		s80.setDoubleValue( 0.0 );
	}
	
	# FLARM
	if( bus_volts > 8 and circuit_breakers.flarm.getBoolValue() ){	# minimal working voltage
		flarm.setDoubleValue( bus_volts );
		load += 0.66 / bus_volts; #FLARM
		load += 0.12 / bus_volts; #FLARM display
	} else {
		flarm.setDoubleValue( 0.0 );
	}
	
	amps.setDoubleValue( load );
	volts.setDoubleValue( bus_volts );
	
	return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	reset_circuit_breakers();
	
	system_updater.enable();
	print("Electrical System initialized");
});

